from datetime import date
from decimal import Decimal
from typing import Dict, List, Optional, Tuple

import attr

from django.db.models import Sum, Min

from staff.map.models import Office
from staff.workspace_management.models import RoomSharePie


@attr.s(auto_attribs=True)
class OfficeAreaData:
    office_id: int
    office_name: str
    office_name_en: str
    rooms_area: Decimal
    office_area: Decimal
    public_area: Decimal


def _office_names(office_ids: List[int]) -> Dict[int, Tuple[str, str]]:
    office_names_qs = (
        Office.objects
        .filter(id__in=office_ids)
        .values('id', 'name', 'name_en')
    )
    return {row['id']: (row['name'], row['name_en']) for row in office_names_qs}


def offices_area(office_id: Optional[int] = None) -> Dict[int, OfficeAreaData]:
    today = date.today()
    qs = (
        RoomSharePie.objects
        .filter(from_date__lte=today, to_date__gt=today)
        .filter(room__floor__office__areas__from_date__lte=today, room__floor__office__areas__to_date__gt=today)
        .values('room__floor__office_id')
        .annotate(rooms_area=Sum('room_area'), office_area=Min('room__floor__office__areas__office_area'))
    )

    if office_id is not None:
        qs = qs.filter(room__floor__office_id=office_id)

    office_names = _office_names([row['room__floor__office_id'] for row in qs])

    result = {}
    for row in qs:
        office_id = row['room__floor__office_id']
        assert office_id is not None
        office_area_data = OfficeAreaData(
            office_id=office_id,
            office_name=office_names[office_id][0],
            office_name_en=office_names[office_id][1],
            rooms_area=row['rooms_area'],
            office_area=row['office_area'],
            public_area=row['office_area'] - row['rooms_area'],
        )
        result[office_area_data.office_id] = office_area_data

    return result
