from datetime import date
from decimal import Decimal
from typing import List, Optional
from uuid import uuid1, UUID

import attr

from staff.workspace_management import entities


@attr.s(auto_attribs=True)
class CreateShare:
    business_unit_id: int
    share_value: Decimal


@attr.s(auto_attribs=True)
class CreateSharePieRequest:
    room_id: int
    author_login: str
    room_area: Decimal
    shares: List[CreateShare]


class CreateSharePieUsecase:
    def __init__(self, repository: entities.Repository, staff: entities.Staff, today: date = None) -> None:
        self._repository = repository
        self._staff = staff
        self._today = today or date.today()
        self.share_pie_id: Optional[UUID] = None

    def create(self, create_share_pie_request: CreateSharePieRequest) -> None:
        if sum([share.share_value for share in create_share_pie_request.shares]) > Decimal(100):
            raise entities.WrongShareSumError('Sum of share more than 100')

        share_pie_id = uuid1()

        shares = [
            entities.Share(share.share_value, share.business_unit_id)
            for share in create_share_pie_request.shares
        ]
        new_pie = entities.RoomSharePie(
            room_share_pie_id=share_pie_id,
            room_id=create_share_pie_request.room_id,
            author_id=self._staff.get_person_id(create_share_pie_request.author_login),
            from_date=self._today,
            to_date=date.max,
            room_area=create_share_pie_request.room_area,
            shares=shares,
        )
        previous_pie = self._repository.last_share_pie_for_room(create_share_pie_request.room_id)
        self._repository.save_share_pie(new_pie)
        if previous_pie:
            previous_pie.to_date = self._today
            self._repository.save_share_pie(previous_pie)
        self.share_pie_id = share_pie_id
