# coding: utf-8
from __future__ import unicode_literals
import six

from .constants import NOTHING


class BaseDataGenerator(object):
    """
        Базовый клас для генератора, который возвращает поток из
        соответствующих данных объектов из базы и объектов из внешнего
        источника,
        а также ключа по которому они сравниваются.
        Так же генератор умеет по ключу получить объект из базы, для выполнения
        над ним операций.
    """
    sync_fields = ()
    diff_fields = ()

    def __iter__(self):
        int_objects = self.get_internal_objects()
        int_objects_dict = {
            self.build_sync_key(item): item for item in int_objects
        }

        for ext_data in self.get_external_objects():
            sync_key = self.build_sync_key(ext_data)
            int_data = int_objects_dict.pop(sync_key, NOTHING)

            yield ext_data, int_data, sync_key

        for sync_key, int_data in six.iteritems(int_objects_dict):
            yield NOTHING, int_data, sync_key

    def get_diff_fields(self):
        return self.diff_fields

    def get_sync_fields(self):
        return self.sync_fields

    def build_sync_key(self, data):
        return tuple(data[f] for f in self.get_sync_fields())

    def get_object(self, sync_key):
        raise NotImplementedError()

    def get_internal_objects(self):
        raise NotImplementedError()

    def get_external_objects(self):
        raise NotImplementedError()

    # bw compatibility
    get_int_objects = get_internal_objects
    ext_data_gen = get_external_objects


class DjangoGenerator(BaseDataGenerator):
    """Django-специфичный генератор"""

    model = None
    queryset = None

    def __init__(self, *args, **kwargs):
        self.value_fields = self.get_sync_fields() + self.get_diff_fields()

    def get_model(self):
        return self.model or self.queryset.model

    def get_queryset(self):
        queryset = self.queryset
        if self.queryset is None:
            queryset = self.get_model().objects.all()
        return queryset

    def get_object(self, sync_key):
        lookup = dict(zip(self.get_sync_fields(), sync_key))
        try:
            obj = self.get_queryset().get(**lookup)
        except self.get_model().DoesNotExist:
            obj = None
        return obj

    def get_internal_objects(self):
        return self.get_queryset().values(*self.value_fields)
