# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging

import six

from .constants import CREATE, UPDATE, DELETE, NOTHING


class BaseDataDiffMerger(object):
    """
        Сравнивает данные из базы и данные пришедшие из внешнего источника.
        Если нет данных из базы - объект создается.
        Если нет внешних данных - объект удаляется.
        Если есть и те и те данные - изменения накатываются на существующий
        объект.
    """

    def __init__(self, data_generator, logger=None, create=False, update=False, delete=False):
        accepted_actions = set()
        if create:
            accepted_actions.add(CREATE)
        if update:
            accepted_actions.add(UPDATE)
        if delete:
            accepted_actions.add(DELETE)
        self.accepted_actions = accepted_actions

        self.created = 0
        self.updated = 0
        self.deleted = 0
        self.errors = 0
        self.skipped = 0
        self.all = 0

        self.data_generator = data_generator

        self.data_generator_name = self.data_generator.__class__.__name__

        if logger is None:
            logger = logging.getLogger(__name__)
        self.logger = logger

    def detect_action(self, ext_data, int_data):
        """
            Определяем действие, которое надо выполнить над объектом.
        """
        if int_data is not NOTHING and ext_data is not NOTHING:
            return UPDATE
        elif ext_data is not NOTHING:
            return CREATE
        else:
            return DELETE

    def create_object(self, ext_data):
        raise NotImplementedError()

    def update_object(self, obj, ext_data):
        raise NotImplementedError()

    def delete_object(self, obj):
        raise NotImplementedError()

    def try_action(self, action, sync_key, diff_data=None, obj=None):
        """
            Пытаемся выполнить действие над объектом.
            Пишем в лог. Считаем сколько каких действий выполнили.
        """
        try:
            if action == CREATE:
                self.create_object(diff_data)
            elif action == UPDATE:
                self.update_object(obj, diff_data)
            else:
                self.delete_object(obj)
        except Exception as e:
            msg = 'Failed to %s %s %s. Exception: %s'
            self.logger.error(msg % (action,
                                     self.data_generator_name,
                                     sync_key,
                                     e))
            self.errors += 1
        else:
            if action == CREATE:
                self.created += 1
            if action == UPDATE:
                self.updated += 1
            if action == DELETE:
                self.deleted += 1

            msg = '%s %s %sd.'
            self.logger.debug(msg, self.data_generator_name, sync_key, action)

    def diff_fields_gen(self, ext_data, int_data):
        for field in self.data_generator.get_diff_fields():
            if field in ext_data:
                ext_value = ext_data.get(field)
                int_value = int_data.get(field)
                if ext_value != int_value:
                    yield field, ext_value

    def diff(self, ext_data, int_data):
        return dict(self.diff_fields_gen(ext_data, int_data))

    def execute(self):
        for ext_data, int_data, sync_key in self.data_generator:
            self.logger.debug('Merging %s with %s by %s', ext_data, int_data, sync_key)
            self.all += 1

            action = self.detect_action(ext_data, int_data)
            self.logger.debug('Gonna %s', action)
            if action not in self.accepted_actions:
                self.logger.debug('%s is not accepted action, so skip')
                self.skipped += 1
                continue

            obj = None
            if action == CREATE:
                self.logger.debug('Gonna create new object')
            else:
                obj = self.data_generator.get_object(sync_key)
                if obj is None:
                    self.skipped += 1
                    self.logger.error('Cannot find object %s %s.', self.data_generator_name, sync_key)
                    continue

            diff_data = None
            if action == UPDATE:
                diff_data = self.diff(ext_data, int_data)
                if not diff_data:
                    self.logger.debug('Object %s is up to date already', sync_key)
                    self.skipped += 1
                    continue
                else:
                    self.logger.debug('Data differs by %s', diff_data)
            elif action == CREATE:
                diff_data = ext_data

            self.try_action(action, sync_key, diff_data=diff_data, obj=obj)

    @property
    def results(self):
        return {
            'created': self.created,
            'updated': self.updated,
            'deleted': self.deleted,
            'errors': self.errors,
            'skipped': self.skipped,
            'all': self.all,
        }


class DjangoDiffMerger(BaseDataDiffMerger):
    def get_model(self):
        return self.data_generator.get_model()

    def set_data(self, obj, data):
        for field, value in six.iteritems(data):
            setattr(obj, field, value)

    def create_object(self, diff_data):
        model_class = self.get_model()
        obj = model_class()
        self.set_data(obj, diff_data)
        obj.save()
        return obj

    def update_object(self, obj, diff_data):
        self.set_data(obj, diff_data)
        obj.save()
        return obj

    def delete_object(self, obj):
        obj.delete()
