from collections import defaultdict

from django.contrib.auth.models import Group
from django_idm_api.hooks import AuthHooks, IntranetHooksMixin, RoleNotFound

from intranet.table_flow.src.rules.models import Rule, RuleUser
from intranet.table_flow.src.users.models import User


class IdmHooks(AuthHooks, IntranetHooksMixin):
    def _get_admin_roles_tree(self):
        return {
            'name': {'ru': 'Роли админки', 'en': 'Admin roles'},
            'roles': {
                'slug': 'role',
                'name': {'ru': 'Роль', 'en': 'Role'},
                'values': {
                    'superuser': {
                        'name': {'ru': 'Суперпользователь', 'en': 'Superuser'},
                        'unique_id': 'superuser',
                    },
                    'rule_editor': {
                        'name': {'ru': 'Редактор правил', 'en': 'Rule editor'},
                        'unique_id': 'rule_editor',
                    },
                },
            },
        }

    def _get_access_to_rules_tree(self):
        return {
            'name': {'ru': 'Доступы до правил', 'en': 'Access to rules'},
            'roles': {
                'slug': 'rule',
                'name': 'Rule',
                'values': {
                    str(rule.id): {
                        'name': rule.slug,
                        'roles': {
                            'slug': 'role',
                            'name': {'ru': 'Роль', 'en': 'Role'},
                            'values': {
                                'rule_administrator': {
                                    'name': {'ru': 'Администратор', 'en': 'Administrator'},
                                    'unique_id': f'rule_admin_{rule.id}',
                                },
                            },
                        },
                    }
                    for rule in Rule.objects.values_list('id', 'slug', named=True)
                },
            },
        }

    def info(self):
        return {
            'code': 0,
            'roles': {
                'slug': 'role_type',
                'name': {'ru': 'Тип роли', 'en': 'Role type'},
                'values': {
                    'admin_roles': self._get_admin_roles_tree(),
                    'access_to_rules': self._get_access_to_rules_tree(),
                },
            },
        }

    def add_role_impl(self, login, role, fields, **kwargs):
        role_name = role.get('role')
        if role_name is None:
            raise RoleNotFound('Role is not not provided')

        user = self._get_user(login)
        if role_name == 'superuser':
            user.is_staff = True
            user.is_superuser = True
            user.save(update_fields=('is_staff', 'is_superuser'))
        elif role_name == 'rule_editor':
            user.is_staff = True
            user.groups.add(Group.objects.get(name='rule_editors'))
            user.save(update_fields=('is_staff',))
        elif role_name == 'rule_administrator':
            try:
                rule = Rule.objects.get(id=role['rule'])
                RuleUser.objects.get_or_create(rule=rule, user=user)
            except Rule.DoesNotExist:
                raise RoleNotFound('Role does not exist: %s' % role)
        else:
            raise RoleNotFound('Role does not exist: %s' % role)

    def remove_role_impl(self, login, role, data, is_fired, **kwargs):
        role_name = role.get('role')
        if role_name is None:
            raise RoleNotFound('Role is not not provided')

        user = self._get_user(login)

        if role_name == 'superuser':
            user.is_superuser = False
            user.save(update_fields=('is_superuser',))
        elif role_name == 'rule_editor':
            user.groups.remove(Group.objects.get(name='rule_editors'))
        elif role_name == 'rule_administrator':
            try:
                rule = Rule.objects.get(id=role['rule'])
                RuleUser.objects.filter(rule=rule, user=user).delete()
            except Rule.DoesNotExist:
                raise RoleNotFound('Role does not exist: %s' % role)
        else:
            raise RoleNotFound('Role does not exist: %s' % role)

    def get_all_roles_impl(self, **kwargs):
        superusers = User.objects.filter(is_superuser=True)
        rule_editors = Group.objects.get(name='rule_editors').user_set.all()
        all_roles = defaultdict(list)
        for user in superusers:
            all_roles[user.username].append({'role': 'superuser', 'role_type': 'admin_roles'})
        for user in rule_editors:
            all_roles[user.username].append({'role': 'rule_editor', 'role_type': 'admin_roles'})
        for rule_id, login in RuleUser.objects.values_list('rule_id', 'user__username'):
            all_roles[login].append({
                'role_type': 'access_to_rules',
                'rule': str(rule_id),
                'role': 'rule_administrator',
            })
        # когда добавим tvm приложения нужно добавить роли для них в all_roles
        return all_roles.items()
