from json import JSONDecodeError

import attr
import requests

from django import forms
from django.conf import settings
from django.contrib import admin
from django.core.exceptions import ValidationError

from intranet.table_flow.src.rules import models, logic
from intranet.table_flow.src.rules.tvm import get_service_ticket_header


class RuleAdminForm(forms.ModelForm):
    class Meta:
        model = models.Rule
        fields = ['slug', 'file', 'json', 'validation_url', 'tvm_id']

    def clean_tvm_id(self):
        value = self.cleaned_data.get('tvm_id')
        if value and str(value) not in settings.TVM_SERVICES:
            raise ValidationError('Invalid or unknown TVM service id. Add it to settings.TVM_SERVICES')
        return value

    def external_validation(self, validation_url, tvm_id, rules):
        headers = get_service_ticket_header(tvm_id)
        response = requests.post(
            validation_url,
            headers=headers,
            json=rules,
            timeout=settings.TF_STAFF_TIMEOUT,
            allow_redirects=False,
        )

        if response.status_code == 404:
            raise ValidationError('Validation URL is not valid')

        if response.status_code == 403:
            raise ValidationError('Has no permissions to access validation URL')

        if 300 <= response.status_code < 400:
            raise ValidationError('Redirects are not allowed on validation URL. Feel free to ask dev team about this')

        if response.status_code != 200:
            try:
                error_dict = response.json()
                messages = error_dict['messages']
                raise ValidationError('; '.join(messages))
            except (JSONDecodeError, KeyError):
                raise ValidationError(response.content)

    def clean(self):
        super().clean()
        try:
            rules_as_json = attr.asdict(logic.parse_file(self.cleaned_data['file']))
            validation_url = self.cleaned_data.get('validation_url')
            if validation_url:
                tvm_id = self.cleaned_data.get('tvm_id')
                if not tvm_id:
                    raise ValidationError('tvm_id should\'t be empty to use validation url')
                self.external_validation(validation_url, tvm_id, rules_as_json)

            self.cleaned_data['json'] = rules_as_json
        except ValueError as e:
            self.cleaned_data['json'] = {}
            raise forms.ValidationError({'file': e})


class RuleAdmin(admin.ModelAdmin):
    form = RuleAdminForm
    list_display = ['slug']
    search_fields = ['slug']

    def get_queryset(self, request):
        return self._filter_rules_by_user(super().get_queryset(request), request.user)

    def save_model(self, request, obj, form, change):
        super().save_model(request, obj, form, change)
        if not change:
            rule_user = models.RuleUser.objects.create(rule=obj, user=request.user)
            rule_user.save()

    @staticmethod
    def _filter_rules_by_user(rules, user):
        if user.is_superuser:
            return rules
        return rules.filter(id__in=models.RuleUser.objects.filter(user_id=user.id).values('rule__id'))


class RuleUserAdmin(admin.ModelAdmin):
    autocomplete_fields = ['rule', 'user']


admin.site.register(models.Rule, RuleAdmin)
admin.site.register(models.RuleUser, RuleUserAdmin)
