import operator
from collections import defaultdict
from typing import Dict, List, Union, Any, Optional

import attr
from django import forms
from distutils.util import strtobool


class Check(dict):
    available_keys = {'lt', 'gt', 'le', 'ge', 'contains', 'ne', 'eq'}
    reverse_order_operators = {'contains'}

    def __setitem__(self, key, value):
        if key not in self.available_keys:
            raise ValueError(f'No check named {key}. Only {self.available_keys} are acceptable.')
        super().__setitem__(key, value)

    def execute(self, values: Union[str, int, List]) -> bool:
        if not isinstance(values, list):
            values = [values]
        for value in values:
            sub_check_passed = True
            # self.items() always is not empty
            # but for empty items - True is more relevant result
            for op_name, check_value in self.items():
                op = getattr(operator, op_name)
                if op_name in self.reverse_order_operators:
                    is_checked = op(check_value, value)
                else:
                    is_checked = op(value, check_value)

                if not is_checked:
                    sub_check_passed = False
            if sub_check_passed:
                return True
        return False


class FieldName(str):
    pass


def cast_field(field_value: Any, field_type: str) -> Union[str, int]:
    field_type_map = {
        'int': int,
        'str': str,
    }
    type_func = field_type_map[field_type]
    return type_func(field_value)


@attr.s(auto_attribs=True)
class Field:
    field_type: str = attr.ib(
        default='str',
        validator=attr.validators.in_(['str', 'int']),
    )

    def as_dict(self):
        return attr.asdict(self)


@attr.s(auto_attribs=True)
class Case:
    checks: Dict[FieldName, Check] = attr.Factory(lambda: defaultdict(Check))
    out: Dict[FieldName, str] = attr.Factory(dict)

    def as_dict(self):
        return {
            'checks': {
                str(fname): dict(check)
                for fname, check in self.checks.items()
            },
            'out': {
                str(fname): value
                for fname, value in self.out.items()
            },
        }

    @classmethod
    def from_dict(cls, inp_dict: dict) -> 'Case':
        case = Case()
        case.out = inp_dict['out']
        for field_name, value_check in inp_dict['checks'].items():
            case.checks[field_name] = Check(value_check)
        return case

    def execute(self, in_values: Dict[str, Any]) -> Optional[Dict[FieldName, str]]:
        for field, check in self.checks.items():
            in_value = in_values.get(field)
            check_passed = check.execute(in_value)

            if not check_passed:
                return
        return self.out


@attr.s(auto_attribs=True)
class Rule:

    in_fields: Dict[FieldName, Field] = attr.Factory(dict)
    out_fields: Dict[FieldName, Field] = attr.Factory(dict)
    cases: List[Case] = attr.Factory(list)

    @classmethod
    def from_dict(cls, inp_dict: dict) -> 'Rule':
        """ No validation included. Only for dicts, created by attr.asdict(Rule)"""
        instance = cls()
        for inst_fields, val_fields in (
            (instance.in_fields, inp_dict['in_fields']),
            (instance.out_fields, inp_dict['out_fields']),
        ):
            for field_name, field in val_fields.items():
                inst_fields[field_name] = Field(field['field_type'])

        instance.cases = [Case.from_dict(case) for case in inp_dict['cases']]
        return instance

    def as_dict(self):
        return {
            'in_fields': {
                str(fname): field.as_dict()
                for fname, field in self.in_fields.items()
            },
            'out_fields': {
                str(fname): field.as_dict()
                for fname, field in self.out_fields.items()
            },
            'cases': [it.as_dict() for it in self.cases]
        }

    def execute(self, in_values: Dict[str, Any]) -> Optional[Dict[FieldName, str]]:
        in_values = in_values.copy()
        fetch_all = strtobool(in_values.pop('fetch_all', ['f'])[0])

        in_values = self.validate_in_values(in_values)

        res = {'results': []} if fetch_all else None

        for case in self.cases:
            result = case.execute(in_values)
            if result is not None:
                if fetch_all:
                    res['results'].append(result)
                else:
                    return result
        return res

    def validate_in_values(self, in_values: Dict[str, Any]):
        errors = {}
        cleaned_data = {}
        for field_name, field in self.in_fields.items():
            try:
                cleaned_data[field_name] = cast_field(in_values[field_name], field.field_type)
            except KeyError:
                errors[field_name] = 'field is required'
            except ValueError as e:
                errors[field_name] = str(e)

        if errors:
            raise forms.ValidationError(errors)

        return cleaned_data
