import re
from typing import List, Any, Dict

from openpyxl import load_workbook

from intranet.table_flow.src.rules import domain_objs


_PATTERNS_REGEXES = {
    'le': re.compile(r'^<= *\d+$'),
    'ge': re.compile(r'^>= *\d+$'),
    'gt': re.compile(r'^> *\d+$'),
    'lt': re.compile(r'^< *\d+$'),
    'ne': re.compile(r'^!= *.+$'),
    'between': re.compile(r'^\d+ *- *\d+$'),
}


def create_check(condition: str, field_type: str) -> domain_objs.Check:
    condition = condition.strip()
    check = domain_objs.Check()
    pattern = None
    for cur_pat, regex in _PATTERNS_REGEXES.items():
        if regex.fullmatch(condition):
            pattern = cur_pat
            break
    if not pattern:
        if ';' in condition:
            check['contains'] = [
                domain_objs.cast_field(it.strip(), field_type)
                for it in condition.split(';')
            ]
        else:
            check['eq'] = domain_objs.cast_field(condition, field_type)
    elif pattern == 'between':
        check['ge'], check['le'] = [
            domain_objs.cast_field(it.strip(), field_type)
            for it in condition.split('-')
        ]
    elif pattern == 'ne':
        check['ne'] = domain_objs.cast_field(condition[2:].strip(), field_type)
    else:
        check[pattern] = domain_objs.cast_field(condition.strip('<>= '), field_type)
    return check


def detect_types(fields: Dict[domain_objs.FieldName, domain_objs.Field], values: Dict[str, Any]):
    for field_name, field in fields.items():
        if field.field_type is not None:
            continue

        field_value = values[field_name]
        value_without_special_chars = re.sub(r'\W+', '', field_value)
        if value_without_special_chars.isdigit():
            field.field_type = 'int'
        else:
            field.field_type = 'str'


def parse_rules(rows: List[List[str]]) -> domain_objs.Rule:
    result = domain_objs.Rule()

    field_names = []
    for column, field_name in enumerate(rows[0], start=1):
        field = domain_objs.Field()
        field_type = None
        if not field_name:
            raise ValueError(f'Field name in column {column} is not defined')

        splitted = field_name.rsplit(':', 1)
        if len(splitted) == 2:
            field_name, field_type = splitted

        if field_name.startswith('in_'):
            field_name = field_name[3:]
            result.in_fields[field_name] = field
            result.in_fields[field_name].field_type = field_type
        elif field_name.startswith('out_'):
            field_name = field_name[4:]
            result.out_fields[field_name] = field
        else:
            raise ValueError(f'Field name must start with in_ or out_, now: {field_name}')

        if field_name in field_names:
            raise ValueError(f'Field names must be unique, field "{field_name}" is ambiguous')

        field_names.append(field_name)

    for i, row in enumerate(rows):
        if i == 0:
            # not slice coz we don't want to copy rows
            continue
        case = domain_objs.Case()
        result.cases.append(case)
        row_dict = dict(zip(field_names, row))
        detect_types(result.in_fields, row_dict)

        for field_name, row_value in row_dict.items():
            row_value = str(row_value)
            if field_name in result.out_fields:
                case.out[field_name] = row_value
                continue

            try:
                check = create_check(row_value, result.in_fields[field_name].field_type)
            except ValueError as e:
                raise ValueError(f'Row {i}, field {field_name}: {e}')
            else:
                case.checks[field_name] = check

    return result


def parse_file(file_obj) -> domain_objs.Rule:
    try:
        wb = load_workbook(file_obj, read_only=True)
    except Exception:
        raise ValueError('Could not parse file. Make sure that you load correct XLSX file.')

    if not len(wb.sheetnames) == 1:
        raise ValueError('File must contain only one sheet.')

    rows = list(wb.active.values)
    if not rows:
        raise ValueError('File is empty.')

    return parse_rules(rows)
