import json
from logging import getLogger

from django import forms
from django.http import JsonResponse
from django.shortcuts import get_object_or_404
from django.views.decorators.http import require_http_methods


from . import models, domain_objs


log = getLogger(__name__)


@require_http_methods(['GET'])
def execute_rule_view(request, slug):
    rule_model = get_object_or_404(models.Rule, slug=slug)
    rule = domain_objs.Rule.from_dict(rule_model.json)

    try:
        result = rule.execute(request.GET)
    except forms.ValidationError as e:
        status_code = 400
        result = e.message_dict
    else:
        status_code = 200
        if result is None:
            log.warning('Rule %s does not match any row', rule_model.id)
            result = {}

    return JsonResponse(result, status=status_code)


@require_http_methods(['POST'])
def execute_bulk_rule_view(request, slug):
    rule_model = get_object_or_404(models.Rule, slug=slug)
    rule = domain_objs.Rule.from_dict(rule_model.json)

    try:
        data = json.loads(request.body)
    except json.JSONDecodeError:
        return JsonResponse({'': 'Invalid input json'}, status=400)

    if 'requests' not in data:
        return JsonResponse({'': 'Missing "requests" property'}, status=400)

    requests = data['requests']

    if not isinstance(requests, list):
        return JsonResponse({'': '"requests" property should be list'}, status=400)

    results = []
    errors = []
    for idx, request_item in enumerate(requests):
        try:
            result = rule.execute(request_item)
        except forms.ValidationError as e:
            errors.append(e.message_dict)
            continue
        else:
            errors.append(dict())

        if result is None:
            log.warning(
                'Rule %s does not match any row for request at index %s',
                rule_model.id,
                idx,
            )
            result = {}

        results.append(result)

    if any(errors):
        return JsonResponse({'errors': errors}, status=400)

    return JsonResponse({'results': results}, status=200)
