import datetime
import logging
from typing import List

import pytz
from django.conf import settings
from ids.registry import registry

from intranet.table_flow.src.users import models


BATCH_SIZE = 250

staff_repo = registry.get_repository(
    'staff', 'person',
    user_agent=settings.TF_USER_AGENT,
    oauth_token=settings.TF_OAUTH_TOKEN,
    timeout=settings.TF_STAFF_TIMEOUT,
)
log = logging.getLogger(__name__)


def get_staff_users() -> List[models.User]:
    lookup = {
        '_sort': 'id',
        '_limit': BATCH_SIZE,
        '_fields': ','.join((
            'id',
            'login',
            'official.is_dismissed',
            'official.join_at',
            'name.first.en',
            'name.last.en',
            'name.first.ru',
            'name.last.ru',
            'work_email',
        )),
    }

    result = []
    query_template = 'id > {}'
    last_id = 0
    has_staff_to_sync = True
    while has_staff_to_sync:
        has_staff_to_sync = False
        lookup['_query'] = query_template.format(last_id)
        persons = _call_api(lookup)
        for person in persons:
            has_staff_to_sync = True
            staff_id = person['id']
            result.append(models.User(
                username=person['login'],
                is_active=not person['official']['is_dismissed'],
                first_name=person['name']['first']['en'],
                last_name=person['name']['last']['en'],
                first_name_ru=person['name']['first']['ru'],
                last_name_ru=person['name']['last']['ru'],
                email=person['work_email'],
                date_joined=_date_to_utc(person['official']['join_at']),
                staff_id=staff_id,
            ))
        if persons:
            last_id = staff_id
    return result


def update_users_info(update_users=False) -> int:
    """ Returns number of created users """
    staff_users = get_staff_users()
    staff_id_to_id = dict(models.User.objects.values_list('staff_id', 'id'))
    to_create = []
    for s_user in staff_users:
        db_id = staff_id_to_id.get(s_user.staff_id)
        if db_id and update_users:
            s_user.id = db_id
            s_user.save()
        elif not db_id:
            to_create.append(s_user)
    models.User.objects.bulk_create(to_create)
    return len(to_create)


def _date_to_utc(str_date) -> datetime.datetime:
    date = datetime.datetime.strptime(str_date, '%Y-%m-%d')
    return pytz.utc.localize(date)


def _call_api(lookup):
    return staff_repo.getiter(lookup=lookup).first_page
