import click
import logging

from aiogram import Bot
from aiogram.utils.exceptions import ChatNotFound, BadRequest

from tasha.config import settings
from tasha.core import dbproxy
from tasha.lib.coro_wrapper import coro_run

from tasha.db.gateways.user import UserGateway

bot = Bot(token=settings.TELEGRAM_TOKEN)
logger = logging.getLogger(__name__)


@click.command('make-user-admin')
@click.option('-u', '--username', 'username', type=str, required=True)
@click.option('-id', '--chat-id', 'chat_id', type=int, required=True)
@click.option('-ns', '--no-super', 'no_superadmin', default=None, required=False)
@coro_run
async def make_user_admin(username: str, chat_id: int, no_superadmin):
    """
        The command promotes a user to admin status, providing
        permissions to invite, restrict users, pin messages and to
        promote users by default. 'no_superadmin' option restricts
        user to promote users, it should be called with True argument.
    """
    db_proxy = dbproxy.DBProxy()
    try:
        await db_proxy.connect()
        async with db_proxy.acquire_connection() as conn:
            user_gw = UserGateway(conn)
            user_accounts = await user_gw.get_users_accounts([username])

        if not user_accounts:
            logger.error("invalid username")
            return

        elif not len(user_accounts):
            logger.error("user has no tg ids")
            return

        account = user_accounts[0]

        try:
            await bot.promote_chat_member(
                chat_id=chat_id, user_id=account.telegram_id,
                can_invite_users=True,
                can_restrict_members=True,
                can_pin_messages=True,
                can_promote_members=None if no_superadmin else True
            )

        except ChatNotFound:
            logger.warning('[%s][%s] chat not found', account.telegram_id, chat_id)
            return

        except BadRequest:
            logger.warning('tasha has no rights to promote user')
            return
    finally:
        await db_proxy.close()
        session = await bot.get_session()
        await session.close()


