import logging
import os
from typing import List, Tuple, Awaitable

import asyncpg
from asyncpg import Connection
from asyncpg.pool import PoolConnectionProxy

from tasha.config import settings
from tasha.constants import ACTION
from tasha.lib import cache_storage

logger = logging.getLogger(__name__)


class GroupNotRegistered(Exception):
    pass


class UserMultipleValue(Exception):
    pass


class TelegramAccountMultipleValue(Exception):
    pass


def _extract_connection(args: tuple, kwargs: dict) -> Tuple[PoolConnectionProxy or None, tuple, dict]:
    """Ищет среди переданных параметров connection, и возвращает его, удалив из параметров"""
    conn = kwargs.pop('connection', None)
    if conn is not None:
        return conn, args, kwargs

    if args and isinstance(args[0], PoolConnectionProxy):
        return args[0], args[1:], kwargs

    return None, args, kwargs


def lazy_connection(async_method):
    """
    Если в параметры декорируемого метода не передать connection явно, то попытаемся на время выполнения метода
    взять новый коннекшн из пулла и затем вернуть его обратно в пулл.
    Идея в том, чтобы можно было извне этого модуля вызывать методы без коннекшена, а внутри между вызовами методов
    DBProxy переиспользовались уже взятые из пулла коннекшены.
    Декорируемый метод должен принимать connection первым параметром либо именованным.
    """

    async def wrapper(self: 'DBProxy', *args, **kwargs):
        connection, args, kwargs = _extract_connection(args, kwargs)
        if connection:
            return await async_method(self, connection, *args, **kwargs)

        async with self.acquire_connection() as connection:
            return await async_method(self, connection, *args, **kwargs)

    return wrapper


class DBProxy(object):

    @classmethod
    async def create(cls):
        instance = cls()
        await instance.connect()
        return instance

    def __init__(self):
        self.connection_pool = None
        self.connection_acquiring_timeout = int(os.getenv('POSTGRES_CONNECTIONS_ACQUIRING_TIMEOUT', 5))

    async def connect(self):
        self.connection_pool = await asyncpg.create_pool(
            user=os.environ['POSTGRES_USER'],
            password=os.environ['POSTGRES_PASSWORD'],
            database=os.environ['POSTGRES_NAME'],
            host=os.environ['POSTGRES_HOST'],
            port=os.environ['POSTGRES_PORT'],
            ssl=os.environ.get('POSTGRES_SSL', 'True') != 'False',
            min_size=int(os.getenv('POSTGRES_CONNECTIONS_POOL_SIZE_MIN', 10)),
            max_size=int(os.getenv('POSTGRES_CONNECTIONS_POOL_SIZE_MAX', 50)),
        )

    async def close(self):
        return self.connection_pool.terminate()

    def acquire_connection(self, timeout: int = None) -> Awaitable[Connection]:
        return self.connection_pool.acquire(timeout=timeout or self.connection_acquiring_timeout)

    @lazy_connection
    async def change_username(self, connection, telegram_id, username, user_is_active):
        await connection.execute(
            '''
            UPDATE tasha_telegramaccount
            SET username = $2
            WHERE telegram_id = $1
            ''', telegram_id, username,
        )
        if user_is_active:
            add_renaming_actions_sql = f'''
                WITH action_values AS(
                    SELECT
                        CURRENT_TIMESTAMP, '{ACTION.USER_CHANGED_USERNAME}', tasha_membership.id
                    FROM tasha_membership
                    JOIN tasha_telegramaccount ON tasha_membership.account_id = tasha_telegramaccount.id
                    WHERE
                        tasha_membership.is_active = TRUE AND tasha_telegramaccount.telegram_id = $1
                )
                INSERT INTO tasha_actions (added, action, membership_id)
                SELECT * FROM action_values
            '''
            await connection.execute(add_renaming_actions_sql, telegram_id)

    @lazy_connection
    async def get_user(self, connection, telegram_id, username, is_bot):
        return self._get_user_and_username(connection, telegram_id, username, is_bot)

    async def _get_user_and_username(self, connection, telegram_id, username, is_bot):
        """
        TODO: разобраться что тут происходит и описать
        """
        row = None
        if username is not None:
            username = username.lower()

        request = '''
            SELECT tasha_user.id as id, tasha_user.is_active as is_active,
                   tasha_telegramaccount.telegram_id as telegram_id,
                   tasha_telegramaccount.is_bot as is_bot,
                   tasha_telegramaccount.id as username_id,
                   tasha_telegramaccount.username as username
              FROM tasha_telegramaccount LEFT OUTER JOIN tasha_user
                ON tasha_telegramaccount.user_id = tasha_user.id
             WHERE tasha_telegramaccount.telegram_id = $1::numeric
        '''
        rows = await connection.fetch(request, telegram_id)
        if rows:
            row = dict(rows[0])
            saved_username = row['username']
            if username and username != saved_username:
                row['username'] = username

                # Проверим, есть ли у нас в базе пользователь с таким username
                select_query = '''
                    SELECT username, telegram_id
                      FROM tasha_telegramaccount
                     WHERE username = $1
                '''
                _rows = await connection.fetch(select_query, username)
                if _rows:
                    _row = dict(_rows[0])
                    if _row['telegram_id'] is None:
                        await self.update_username(telegram_id, username, connection=connection)
                    else:
                        # ToDo: понять как обрабатывать такой случай
                        logger.error(
                            'telegram_id `%s` have username `%s`, but in db this username have telegram_id `%s`',
                            telegram_id, username, _row['telegram_id']
                        )
                        raise TelegramAccountMultipleValue()

                else:
                    await self.change_username(telegram_id, username, row['is_active'], connection=connection)

        elif username:
            request += ' OR tasha_telegramaccount.username = $2'
            rows = await connection.fetch(request, telegram_id, username)
            if rows:
                await connection.execute(
                    '''
                    UPDATE tasha_telegramaccount
                    SET telegram_id = $2
                    WHERE username = $1
                    ''', username, telegram_id
                )
                row = dict(rows[0])
                row['telegram_id'] = telegram_id

        if row is None:
            logger.info('username for %s:%s not found', username, telegram_id)
            rows = await connection.fetch(
                '''
                    INSERT INTO tasha_telegramaccount
                        (user_id, username, telegram_id, is_bot)
                    VALUES
                        (null, $1, $2::numeric, $3)
                    RETURNING
                        null as id, null as is_active, telegram_id, is_bot, id as username_id, username
                ''', username, telegram_id, is_bot
            )
            row = dict(rows[0])
        # слишком много сообщений этого типа в логах
        # logger.info('user for %s:%s -> %s', username, telegram_id, row)
        if row['is_bot'] != is_bot:
            await connection.execute(
                '''
                UPDATE tasha_telegramaccount
                SET is_bot = $1
                WHERE id = $2
                ''', is_bot, row['username_id']
            )
            row['is_bot'] = is_bot

        return row

    @lazy_connection
    async def get_subbot_ids(self, connection):
        select_query = '''
            SELECT telegram_id
            FROM tasha_telegramaccount
            WHERE is_tasha = TRUE
        '''
        return await connection.fetch(select_query)

    @lazy_connection
    async def get_subbots_usernames(self, connection):
        rows = await connection.fetch('''
            SELECT username
            FROM tasha_telegramaccount
            WHERE is_tasha = TRUE
        ''')
        return {row['username'] for row in rows}

    @lazy_connection
    async def get_group(self, connection, chat_id, ignore_deactivated=False):
        query = '''
            SELECT *
            FROM tasha_tggroupinfo
            WHERE telegram_id = $1::numeric
        '''
        if ignore_deactivated:
            query += ' AND NOT deactivated'

        rows = await connection.fetch(query, chat_id)

        if not rows:
            raise GroupNotRegistered('Group with telegram_id %s not found' % chat_id)

        return rows[0]

    @lazy_connection
    async def get_available_chats(self, connection, telegram_id: int, is_admin: bool = None):
        """
        Возвращает `telegram_id`, `chat_title`, `only_yandex` всех чатов, где состоит указанный телеграм-аккаунт
        Если указать is_admin, пофильтрует мембершипы по этому полю
        """
        is_admin_conditions = f'AND tasha_membership.is_admin = {is_admin}' if is_admin is not None else ''
        select_query = f'''
            SELECT tgroup.telegram_id, tgroup.title, tgroup.only_yandex
            FROM
                tasha_membership as mm
                INNER JOIN tasha_tggroupinfo as tgroup ON mm.group_id = tgroup.id
                INNER JOIN tasha_telegramaccount ON mm.account_id = tasha_telegramaccount.id
            WHERE
                tgroup.deactivated = FALSE
                AND mm.is_active = TRUE
                AND tasha_telegramaccount.telegram_id = $1::numeric
                {is_admin_conditions}
        '''
        return await connection.fetch(select_query, telegram_id)

    @lazy_connection
    async def get_active_chats_for(self, connection, username: str, god_bot_number: int = None):
        """
        Возвращает `telegram_id` всех чатов, где состоит указанный телеграм-аккаунт
        Если указать god_bot_number - фильтрует по воркеру.
        Возвращает чаты в порядке возрастания last_successful_scan
        """
        is_god_bot_conditions = f'AND tasha_tggroupinfo.god_bot_number = {god_bot_number}' if god_bot_number is not None else ''
        select_query = f'''
            SELECT DISTINCT tasha_tggroupinfo.telegram_id, tasha_tggroupinfo.last_successful_scan
            FROM
                tasha_tggroupinfo
                INNER JOIN tasha_membership ON (tasha_membership.group_id = tasha_tggroupinfo.id)
                INNER JOIN tasha_telegramaccount ON (tasha_membership.account_id = tasha_telegramaccount.id)
            WHERE
                tasha_tggroupinfo.deactivated = FALSE
                AND tasha_membership.is_active = TRUE
                AND tasha_telegramaccount.username = $1
                {is_god_bot_conditions}
            ORDER BY last_successful_scan NULLS FIRST
        '''

        rows = await connection.fetch(select_query, username)

        return rows

    @lazy_connection
    async def get_all_active_groups_with_true_bot(self, connection, bot_username):
        return await connection.fetch('''
            SELECT DISTINCT tasha_tggroupinfo.telegram_id
            FROM
                tasha_tggroupinfo
                INNER JOIN tasha_membership ON (tasha_membership.group_id = tasha_tggroupinfo.id)
                INNER JOIN tasha_telegramaccount ON (tasha_membership.account_id = tasha_telegramaccount.id)
            WHERE
                tasha_tggroupinfo.deactivated = FALSE
                AND tasha_membership.is_active = TRUE
                AND tasha_telegramaccount.username = $1
        ''', bot_username)

    @lazy_connection
    async def get_users_to_unblock(self, connection, bot_account_id=None, god_bot_number: int = None):
        select_sql = '''
            SELECT groups.telegram_id as chat_id, groups.title as chat_title, accounts.telegram_id, accounts.username, actions.membership_id
            FROM
                (
                    SELECT act.membership_id, MAX(act.added) as added
                    FROM tasha_actions act
                    WHERE
                        act.group_id IS NULL
                        AND act.membership_id IN (
                            SELECT mbs.id
                            FROM
                                tasha_membership mbs
                                INNER JOIN tasha_tggroupinfo groups on mbs.group_id = groups.id
                            WHERE
                                groups.deactivated = FALSE
                                {god_bot_number_condition}
                        )
                    GROUP BY act.membership_id
                ) as latest_actions
                INNER JOIN tasha_actions actions USING (membership_id, added)
                INNER JOIN tasha_membership memberships ON (actions.membership_id = memberships.id)
                INNER JOIN tasha_tggroupinfo groups ON (memberships.group_id = groups.id)
                INNER JOIN tasha_telegramaccount accounts ON (memberships.account_id = accounts.id)
                INNER JOIN tasha_user users ON (accounts.user_id = users.id)
            WHERE
                actions.action = ANY($1::varchar[])
                AND groups.deactivated = FALSE
                AND users.is_active = TRUE
                AND groups.id IN (
                    SELECT groups.id
                    FROM
                        tasha_tggroupinfo groups
                        INNER JOIN tasha_membership memberships ON (memberships.group_id = groups.id)
                    WHERE
                        groups.deactivated = FALSE
                        AND memberships.is_active = TRUE
                        AND memberships.account_id = $2
                        {god_bot_number_condition}
                )
        '''.format(
            god_bot_number_condition='AND groups.god_bot_number = $3' if god_bot_number else ''
        )
        params = [ACTION.user_kicked - {ACTION.USER_BANNED_BY_USER}, bot_account_id or settings.TELEGRAM_BOT_ACCOUNT_ID]
        if god_bot_number:
            params.append(god_bot_number)
        return await connection.fetch(select_sql, *params)

    @lazy_connection
    async def get_participants_count(self, connection, chat_id):
        return (await connection.fetch('''
            SELECT COUNT(*)
            FROM
                tasha_membership
                INNER JOIN tasha_tggroupinfo ON (tasha_membership.group_id = tasha_tggroupinfo.id)
            WHERE tasha_tggroupinfo.telegram_id = $1
            AND tasha_membership.is_active = TRUE
        ''', chat_id))[0]['count']

    @lazy_connection
    async def get_chat_participants(self, connection, chat_id, to_kick):
        to_kick_conditions = '' if not to_kick else """
                AND (tasha_user.is_active = FALSE OR tasha_telegramaccount.user_id IS NULL)
                AND tasha_telegramaccount.is_bot = FALSE
        """
        return await connection.fetch('''
            SELECT tasha_telegramaccount.username, tasha_telegramaccount.telegram_id, tasha_telegramaccount.is_bot
            FROM
                tasha_membership
                INNER JOIN tasha_tggroupinfo ON (tasha_membership.group_id = tasha_tggroupinfo.id)
                INNER JOIN tasha_telegramaccount ON (tasha_membership.account_id = tasha_telegramaccount.id)
                LEFT OUTER JOIN tasha_user ON (tasha_telegramaccount.user_id = tasha_user.id)
            WHERE
                tasha_tggroupinfo.telegram_id = $1
                AND tasha_membership.is_active = TRUE
                AND tasha_tggroupinfo.deactivated = FALSE
                {}
        '''.format(to_kick_conditions), chat_id)

    @lazy_connection
    async def deactivate(self, connection, chat_id):
        await connection.execute('''
            UPDATE tasha_tggroupinfo
            SET deactivated = TRUE
            WHERE telegram_id = $1
        ''', chat_id)

    @cache_storage.memoize_decorator_async(fresh_time=settings.TELEGRAM_TO_PG_CACHE_TIMEOUT)
    @lazy_connection
    async def is_valid_user(self, connection, user_id, username, is_bot):
        if username and (username.lower() in settings.TG_WHITELIST):
            return True

        user = await self._get_user_and_username(connection, user_id, username, is_bot)
        return user['is_active'] or user['is_bot']

    @lazy_connection
    async def register_enter(self, connection, chat_id, user_id, username, is_bot, action):
        # Получаем юзера и группу
        # Перед register_enter должна быть проверка на существование юзера в базе, так что здесь не требуется
        user = await self._get_user_and_username(connection, user_id, username, is_bot)
        username_id = user['username_id']
        is_active = user['is_active']

        group_id = (await self.get_group(chat_id, connection=connection))['id']

        membership_id = (await connection.fetch('''
            INSERT INTO tasha_membership (account_id, group_id, is_active)
            VALUES ($1::numeric, $2::numeric, true)
            ON CONFLICT (account_id, group_id)
            DO UPDATE SET is_active = TRUE, first_notified_from_email = NULL
            RETURNING id
        ''', username_id, group_id))[0]['id']

        await self.add_action_to_membership(action, connection=connection, membership_id=membership_id)
        return membership_id, is_active

    @lazy_connection
    async def register_message(self, connection, user_id, chat_id):
        insert_query = '''
            INSERT INTO tasha_tgmessage (added, user_telegram_id, group_telegram_id)
            VALUES (now(), $1::bigint, $2::bigint)
        '''
        params = (user_id, chat_id)
        await connection.fetch(insert_query, *params)

    @lazy_connection
    async def register_email(self, connection, not_kicked_users, chat_id):
        for user in not_kicked_users:
            try:
                user_id = user['id']
            except TypeError:
                user_id = user.id

            logger.info('Adding action EMAIL for chat_telegram_id %s telegram_id %s', chat_id, user_id)
            await self.add_action_to_membership(
                connection=connection,
                action=ACTION.EMAIL,
                telegram_id=user_id,
                chat_telegram_id=chat_id,
            )
            await self.add_notify_date_to_membership(connection, telegram_id=user_id, chat_telegram_id=chat_id)

    @lazy_connection
    async def register_enter_if_no_membership_active(self, connection, chat_id, user_id, username, is_bot, is_admin):
        account_id = (await self._get_user_and_username(connection, user_id, username, is_bot))['username_id']
        try:
            group_id = (await self.get_group(chat_id, connection=connection))['id']
        except GroupNotRegistered:
            assert False, 'Group not registered but saved in db'

        rows = await connection.fetch('''
              SELECT is_active, is_admin, id
              FROM tasha_membership
              WHERE account_id = $1::numeric AND group_id = $2::numeric
        ''', account_id, group_id)

        actions = []
        is_current_admin = False
        is_current_active = False
        if rows:
            is_current_admin = rows[0]['is_admin']
            is_current_active = rows[0]['is_active']
            if is_current_active and is_current_admin == is_admin:
                # Активное членство уже есть, ничего не меняем
                return

        if not rows or not is_current_active:
            actions.append(ACTION.USER_DETECTED_BY_SUBBOT)
        if is_current_admin != is_admin:
            actions.append(ACTION.USER_GRANTED_ADMIN_RIGHTS if is_admin else ACTION.USER_LOST_ADMIN_RIGHTS)

        await connection.execute('''
              INSERT INTO tasha_membership (account_id, group_id, is_active, is_admin)
              VALUES ($1::numeric, $2::numeric, true, $3)
              ON CONFLICT (account_id, group_id)
              DO UPDATE SET is_active = TRUE, is_admin = $3, first_notified_from_email = NULL
          ''', account_id, group_id, is_admin)

        for action in actions:
            await self.add_action_to_membership(
                connection=connection,
                action=action,
                account_id=account_id,
                group_id=group_id,
            )

    @lazy_connection
    async def add_action_to_membership(self, connection, action, membership_id=None, account_id=None, group_id=None,
                                       telegram_id=None, chat_telegram_id=None):
        if membership_id is not None:
            await connection.execute('''
                INSERT INTO tasha_actions (added, action, membership_id)
                VALUES (CURRENT_TIMESTAMP, $1, $2::numeric)
            ''', action, membership_id)
        elif account_id is not None and group_id is not None:
            await connection.execute('''
                INSERT INTO tasha_actions (added, action, membership_id)
                VALUES (CURRENT_TIMESTAMP, $1,
                    (SELECT id FROM tasha_membership WHERE account_id = $2::numeric AND group_id = $3::numeric))
            ''', action, account_id, group_id)
        elif telegram_id is not None and chat_telegram_id is not None:
            await connection.execute('''
                INSERT INTO tasha_actions (added, action, membership_id)
                VALUES (
                    CURRENT_TIMESTAMP, $1, (
                        SELECT tasha_membership.id
                        FROM tasha_membership
                        JOIN tasha_tggroupinfo ON tasha_membership.group_id = tasha_tggroupinfo.id
                        JOIN tasha_telegramaccount ON tasha_membership.account_id = tasha_telegramaccount.id
                        WHERE tasha_telegramaccount.telegram_id = $2 AND tasha_tggroupinfo.telegram_id = $3
                    )
                )
            ''', action, telegram_id, chat_telegram_id)
        else:
            raise AssertionError('Wrong params')

    @lazy_connection
    async def register_exit(self, connection, chat_id, user_id, username, is_bot, action):
        """
        Деактивирует соответствующий мембершип
        """
        username_id = (await self._get_user_and_username(connection, user_id, username, is_bot))['username_id']

        try:
            group = (await self.get_group(chat_id, connection=connection))['id']
        except GroupNotRegistered:
            logger.info('User %s:%s is left from unregistered group %s', username, user_id, chat_id)
            return
        except UserMultipleValue:
            logger.info('username `%s` have telegram_id %s, but they have different users' % (username, user_id))
            return

        await connection.execute('''
              INSERT INTO tasha_membership (account_id, group_id, is_active)
              VALUES ($1::numeric, $2::numeric, false)
              ON CONFLICT (account_id, group_id)
              DO UPDATE SET is_active = false
        ''', username_id, group)
        # Теперь вешаем экшн
        logger.info('[%s] create action %s for %s:%s', chat_id, action, username, user_id)
        await self.add_action_to_membership(
            connection=connection,
            action=action,
            account_id=username_id,
            group_id=group,
        )

    @lazy_connection
    async def get_chat_admins_emails(self, connection, chat_id) -> List[str]:
        SELECT_ADMINS_EMAILS_QUERY = '''
            SELECT tasha_user.email
            FROM tasha_membership
                JOIN tasha_tggroupinfo ON (tasha_membership.group_id = tasha_tggroupinfo.id)
                JOIN tasha_telegramaccount ON (tasha_membership.account_id = tasha_telegramaccount.id)
                JOIN tasha_user ON (tasha_telegramaccount.user_id = tasha_user.id)
            WHERE
                tasha_tggroupinfo.telegram_id = $1
                AND tasha_membership.is_active = TRUE
                AND tasha_tggroupinfo.deactivated = FALSE
                AND tasha_user.is_active = TRUE
                AND tasha_user.email IS NOT NULL
                AND tasha_membership.is_admin = TRUE
                AND NOT tasha_user.username = 'robot-uhura'
        '''
        return [x['email'] for x in await connection.fetch(SELECT_ADMINS_EMAILS_QUERY, chat_id)]

    @lazy_connection
    async def add_notify_date_to_membership(self, connection, telegram_id, chat_telegram_id):
        SELECT_QUERY = '''
            SELECT tasha_membership.id, tasha_membership.first_notified_from_email
              FROM tasha_membership
              JOIN tasha_telegramaccount ON tasha_membership.account_id = tasha_telegramaccount.id
              JOIN tasha_tggroupinfo ON tasha_membership.group_id = tasha_tggroupinfo.id
             WHERE tasha_telegramaccount.telegram_id = $1 AND
                   tasha_tggroupinfo.telegram_id = $2
        '''

        UPDATE_QUERY = '''
            UPDATE tasha_membership SET first_notified_from_email = CURRENT_TIMESTAMP
             WHERE tasha_membership.id = $1
        '''

        pk, current_value = (await connection.fetch(SELECT_QUERY, telegram_id, chat_telegram_id))[0]
        if current_value is None:
            await connection.execute(UPDATE_QUERY, pk)

    @lazy_connection
    async def update_last_successful_scan(self, connection, chat_id):
        await connection.execute('''
            UPDATE tasha_tggroupinfo
            SET last_successful_scan = CURRENT_TIMESTAMP
            WHERE telegram_id = $1
        ''', chat_id)

    @lazy_connection
    async def update_username(self, connection, telegram_id, username):
        # Проверим, что для текущего username нет чатов
        ASSERT_MEMBERSHIP_QUERY = '''
            SELECT COUNT(*) as count
              FROM tasha_membership
              JOIN tasha_telegramaccount
                ON tasha_membership.account_id = tasha_telegramaccount.id
             WHERE tasha_telegramaccount.username = $1
        '''

        SELECT_USER_FROM_USERNAME_QUERY = '''
            SELECT user_id
              FROM tasha_telegramaccount
             WHERE username = $1
        '''

        SELECT_USER_FROM_TELEGRAM_ID_QUERY = '''
            SELECT user_id
              FROM tasha_telegramaccount
             WHERE telegram_id = $1
        '''

        DELETE_QUERY = '''
            DELETE FROM tasha_telegramaccount
             WHERE username = $1;
        '''

        UPDATE_QUERY = '''
            UPDATE tasha_telegramaccount
               SET username = $1, user_id = $2
             WHERE telegram_id = $3;
        '''

        count = (await connection.fetch(ASSERT_MEMBERSHIP_QUERY, username))[0]['count']
        assert count == 0, f'username `{username}` with no telegram_id have {count} memberships'
        # Получим пользователя, привязанного к username
        username_user = (await connection.fetch(SELECT_USER_FROM_USERNAME_QUERY, username))[0]['user_id']
        telegram_id_user = (await connection.fetch(SELECT_USER_FROM_TELEGRAM_ID_QUERY, telegram_id))[0]['user_id']
        if username_user is not None and telegram_id_user is not None and username_user != telegram_id_user:
            logger.error(
                'username `%s` have telegram_id %s, but they have different users (%s, %s)',
                username, telegram_id, username_user, telegram_id_user
            )
            raise UserMultipleValue()

        # Удалим username и обновим данные
        user_id = username_user or telegram_id_user
        async with connection.transaction():
            await connection.execute(DELETE_QUERY, username)
            await connection.execute(UPDATE_QUERY, username, user_id, telegram_id)

    @lazy_connection
    async def update_admin_status(self, connection, chat_id: int, telegram_id: int, is_admin: bool):
        # Проверим, что для текущего username нет чатов
        SELECT_MEMBERSHIP_QUERY = '''
                SELECT mbr.id as id
                  FROM tasha_membership as mbr
                    JOIN tasha_tggroupinfo as tgr ON mbr.group_id = tgr.id
                    JOIN tasha_telegramaccount acc on mbr.account_id = acc.id
                  WHERE
                        tgr.telegram_id = $1
                    AND acc.telegram_id = $2
            '''

        UPDATE_QUERY = '''
                UPDATE tasha_membership
                   SET is_admin = $2
                 WHERE id = $1;
            '''

        rows = (await connection.fetch(SELECT_MEMBERSHIP_QUERY, chat_id, telegram_id))[0]
        if not rows:
            logger.info(
                '[%s] membership for %s not found',
                chat_id, telegram_id,
            )
            return
        membership_id = rows[0].id
        await connection.execute(UPDATE_QUERY, membership_id, is_admin)

    @lazy_connection
    async def register_new_chat(
            self, connection,
            chat_telegram_id: int,
            chat_title: str,
            chat_members_count: int,
            chat_type: str,
            bot_telegram_id: int,
            bot_is_admin: bool,
            god_bot_number: int = None,
    ) -> Tuple[int, int]:
        """
        Создаёт новую группу.
        И добавляет мембершип бота в ней.
        Вернёт id группы и мембершипа.
        """

        create_group_sql = '''
            INSERT INTO tasha_tggroupinfo
                (
                    telegram_id, deactivated, title, count_from_api,
                    should_add_slack_migration_bot, slack_migration_bot_added,
                    chat_type, god_bot_number, auto_approve, only_yandex
                )
            VALUES
                ($1, false, $2, $3, false, false, $4, $5, false, false)
            ON CONFLICT (telegram_id)
            DO UPDATE SET
                deactivated = false, title = $2,  count_from_api = $3, chat_type = $4, god_bot_number = $5
            RETURNING id;
        '''
        add_tasha_to_group_sql = '''
            INSERT INTO tasha_membership (account_id, group_id, is_active, is_admin)
            VALUES (
                (SELECT id FROM tasha_telegramaccount WHERE telegram_id = $1 LIMIT 1),
                $2::numeric,
                true,
                $3
            )
            ON CONFLICT (account_id, group_id)
            DO UPDATE SET
                is_active = TRUE, is_admin = $3, first_notified_from_email = NULL
            RETURNING id;
        '''

        async with connection.transaction():
            params = (chat_telegram_id, chat_title, chat_members_count, chat_type, god_bot_number)
            result = await connection.fetch(create_group_sql, *params)
            created_group_id = result[0]['id']

            params = (bot_telegram_id, created_group_id, bot_is_admin)
            result = await connection.fetch(add_tasha_to_group_sql, *params)
            created_membership_id = result[0]['id']

            await self.add_action_to_membership(
                connection=connection,
                action=ACTION.BOT_DETECTED_NEW_CHAT,
                membership_id=created_membership_id,
            )
            return created_group_id, created_membership_id

    @lazy_connection
    async def get_chats_with_true_bot(self, connection, truebot_telegram_username: str):
        rows = await connection.fetch('''
            SELECT groups.telegram_id as telegram_id
              FROM tasha_tggroupinfo groups
              JOIN tasha_membership mbs ON (groups.id = mbs.group_id)
              JOIN tasha_telegramaccount accounts ON (mbs.account_id = accounts.id)
             WHERE accounts.username = $1 AND mbs.is_active
        ''', truebot_telegram_username)
        return rows

    @lazy_connection
    async def get_chat_whitelist_users(self, connection, chat_telegram_id: int):
        rows = await connection.fetch('''
                SELECT acc.telegram_id as telegram_id, acc.username as username
                FROM tasha_whitelist_entry whl
                  INNER JOIN tasha_tggroupinfo gr on whl.group_id = gr.id
                  INNER JOIN tasha_telegramaccount acc on whl.user_account_id = acc.id
                WHERE gr.telegram_id = $1 AND whl.is_active = TRUE
            ''', chat_telegram_id)
        return rows

    @lazy_connection
    async def get_chats_to_add_slack_bot_to(self, connection):
        select_query = '''
            SELECT g.telegram_id as telegram_id
              FROM tasha_tggroupinfo g
             WHERE
                g.should_add_slack_migration_bot
                AND NOT g.slack_migration_bot_added
                AND NOT g.deactivated
        '''
        return await connection.fetch(select_query)

    @lazy_connection
    async def get_user_chats(self, connection, username):
        select_query = '''
                SELECT DISTINCT tasha_tggroupinfo.telegram_id
                FROM
                    tasha_tggroupinfo
                    INNER JOIN tasha_membership ON (tasha_membership.group_id = tasha_tggroupinfo.id)
                    INNER JOIN tasha_telegramaccount ON (tasha_membership.account_id = tasha_telegramaccount.id)
                    INNER JOIN tasha_user tu on tu.id = tasha_telegramaccount.user_id
                WHERE
                    tasha_tggroupinfo.deactivated = FALSE
                    AND tasha_membership.is_active = TRUE
                    AND tu.username = $1
            '''
        return await connection.fetch(select_query, username)

    @lazy_connection
    async def get_user_acc(self, connection, username: str):
        select_query = '''
                    SELECT *
                    FROM tasha_telegramaccount as acc
                    INNER JOIN tasha_user tu on tu.id = acc.user_id
                    WHERE tu.username = $1
                '''
        rows = await connection.fetch(select_query, username)
        return rows[0] if rows else None

    @lazy_connection
    async def export_groups(self, connection):
        select_query = '''
                    SELECT
                           g.telegram_id as telegram,
                           g.title as title,
                           g.chat_type as type,
                           g.last_successful_scan as last_scan,
                           array_agg(u.username) FILTER ( WHERE not mm.is_admin ) as members,
                           array_agg(u.username) FILTER ( WHERE mm.is_admin ) as admins
                    FROM tasha_membership as mm
                    INNER JOIN tasha_telegramaccount acc on acc.id = mm.account_id
                    INNER JOIN tasha_user u on u.id = acc.user_id
                    INNER JOIN tasha_tggroupinfo g on g.id = mm.group_id
                    WHERE
                          g.deactivated = False
                        AND mm.is_active = True
                    GROUP BY g.telegram_id, g.title, g.chat_type, g.last_successful_scan
                '''
        rows = await connection.fetch(select_query)
        return rows

    @lazy_connection
    async def get_top_channels(self, connection, channels):
        """
        Getting users from given channels for further intersection cnt.
        Ordered by title to provide the correct view via YT
        """
        channels = channels
        select_query = '''
                    SELECT
                            g.title as title,
                            g.telegram_id,
                            g.chat_type,
                            array_agg(mm.account_id) FILTER ( WHERE mm.is_active ) as members,
                            count(mm.account_id) FILTER (WHERE mm.is_active) as cnt_members
                    FROM tasha_membership as mm
                    INNER JOIN tasha_tggroupinfo g on g.id = mm.group_id
                    WHERE
                            g.telegram_id = ANY($1::numeric[])
                    GROUP BY g.telegram_id, g.title, g.chat_type
                    ORDER BY g.title ASC
            '''
        rows = await connection.fetch(select_query, channels.values())
        return rows

    @lazy_connection
    async def get_staff_usernames(self, connection, ids):
        """
        Getting staff usernames having
        ids in tasha_membership
        """
        select_query = '''
                    SELECT tu.username as staff_username
                    FROM tasha_telegramaccount as acc
                    INNER JOIN tasha_user tu on tu.id = acc.user_id
                    INNER JOIN tasha_membership mm on acc.id = mm.account_id
                    WHERE
                            mm.account_id = ANY($1::numeric[])
                '''
        rows = await connection.fetch(select_query, ids)
        return rows
