import logging
import os
from dataclasses import dataclass
from datetime import timedelta, datetime
from typing import Iterable

from aiogram.types import Chat, User
from aiogram.utils.exceptions import BotKicked, ChatNotFound, MigrateToChat, Unauthorized, BadRequest, \
    ChatAdminRequired, UserIsAnAdministratorOfTheChat, CantRestrictChatOwner, NotEnoughRightsToRestrict
from arq import Retry
from cachetools import TTLCache
from pyrogram import types
from pyrogram.errors import FloodWait
from sentry_sdk import capture_exception, capture_message

from tasha.config import settings
from tasha.constants import ACTION
from tasha.core import TashaUnit
from tasha.core.models.group import Group
from tasha.core.services import UserService, MailService
from tasha.core.services.base import BaseService
from tasha.db.gateways.chat import DbParticipant
from tasha.external.telegram import TelegramClient
from tasha.utils import is_in_whitelist

logger = logging.getLogger(__name__)

SMALL_CHAT_SIZE = 10
SMALL_CHAT_KICK_RATIO_THRESHOLD = os.environ.get('SMALL_CHAT_KICK_RATIO_THRESHOLD', 0.7)
KICK_RATIO_THRESHOLD = os.environ.get('KICK_RATIO_THRESHOLD', 0.2)
CHATS_BATCH_SIZE = int(os.environ.get('CHATS_BATCH_SIZE', 10))

SYNC_CACHE = TTLCache(maxsize=100000, ttl=timedelta(hours=12), timer=datetime.now)


@dataclass
class ChatScanStat:
    telegram_id: int
    tasha_id: int = None
    title: str = None

    tasha_members_count: int = None
    telegram_members_count: int = None
    participants_count: int = None

    checked_count: int = None
    aliens_count: int = None


def sync_user_cache_key():
    pass


class ChatScanService(BaseService):
    def __init__(self, tu: TashaUnit):
        self.user_srv = UserService(tu)
        self.mail_srv = MailService(tu)
        super().__init__(tu)

    async def scan(self, chat_id: int, client: TelegramClient, no_retry: bool):
        """
        Сканирует чат и проверяет что все участники чата валидные пользователи
        """
        stat = ChatScanStat(telegram_id=chat_id)
        members_ids = set()
        aliens = []
        not_kicked = []
        too_many = False

        try:
            chat = await self._tu.bot.get_chat(chat_id=chat_id)
        except BotKicked:
            logger.warning('[%s] tasha was kicked from chat. deactivate it', chat_id)
            await self._tu.chat.deactivate(chat_id)
            return
        except ChatNotFound:
            logger.warning('[%s] this chat not found. deactivate it', chat_id)
            await self._tu.chat.deactivate(chat_id)
            return
        except Unauthorized:
            logger.warning('[%s] this chat access denied. deactivate it', chat_id)
            await self._tu.chat.deactivate(chat_id)
            return
        except Exception as err:
            logger.exception('[%s] Fetching group  raised exception %s: %s',
                             chat_id, err.__class__.__name__, err,
                             exc_info=err)
            return

        stat.title = chat.title
        stat.tasha_members_count = await self._tu.chat.get_members_count(chat_id=chat_id)

        group = await self._tu.chat.get(chat_id=chat_id)
        await self._sync_group(chat, group)

        try:
            members_count = await chat.get_member_count()
        except MigrateToChat:
            logger.warning('[%s] chat migrated. deactivate it', chat_id)
            await self._tu.chat.deactivate(chat_id)
            return
        except BotKicked:
            logger.warning('[%s] chat forbitten. deactivate it', chat_id)
            await self._tu.chat.deactivate(chat_id)
            return
        except Unauthorized:
            logger.warning('[%s] this chat access denied. deactivate it', chat_id)
            await self._tu.chat.deactivate(chat_id)
            return
        except Exception as err:
            logger.exception('[%s] Fetching members raised exception %s: %s',
                             chat_id, err.__class__.__name__, err,
                             exc_info=err)
            return

        try:
            chat_admins = await client.get_admins(chat_id)
        except FloodWait:
            raise Retry(defer=30)
        # except Unauthorized:
        #     logger.warning('[%s] this chat access denied get admins', chat_id)
        #     chat_admins = []
        except Exception as err:
            logger.exception('[%s] chat.get_administrators() raised exception %s: %s',
                             chat_id, err.__class__.__name__, err,
                             exc_info=err)
            chat_admins = []

        stat.telegram_members_count = members_count

        try:
            tg_participants = await client.get_chat_members(chat_id)
        except Exception as err:
            capture_exception(err)
            logger.exception('[%s] telegram lib exception %s: %s',
                             chat_id, err.__class__.__name__, err,
                             exc_info=err)
            if not no_retry:
                raise Retry(defer=30)
            else:
                tg_participants = []

        stat.participants_count = len(tg_participants)

        stat.tasha_id = group.id

        db_whitelist = await self._tu.whitelist.get_chat_whitelist_users(chat_id=chat_id)
        db_participants = await self._tu.chat.get_participants(chat_id=chat_id)
        admins_ids = {a.user.id for a in chat_admins}

        all_participants = await self._merge_participants(chat, tg_participants, db_participants)

        db_participants_map = {p.telegram_id: p for p in db_participants}
        for participant in all_participants:
            # TODO: это че вообще?
            if not participant.id:
                logger.error('Что с ним не так? %s', participant)
                capture_message('Что с ним не так? %s' % participant, 'WARNING')
                continue

            is_admin = participant.id in admins_ids
            # Мы не знаем такого пользователя
            if participant.id not in db_participants_map:
                unknown_participant = await self._register_enter(
                    group=group,
                    participant=participant,
                    is_admin=is_admin
                )
                db_participants_map[participant.id] = unknown_participant

            db_participant = db_participants_map[participant.id]
            is_valid_user = await self._is_valid_user(group, db_participant, participant, db_whitelist)

            if not is_valid_user:
                aliens.append((participant, db_participant))
            else:
                await self._sync_user(group, participant, is_admin)
            members_ids.add(participant.id)

        stat.checked_count = len(members_ids)
        stat.aliens_count = len(aliens)

        if aliens:
            logger.warning(
                '[%s] aliens for chat `%s`: %s',
                chat.id, chat.title,
                [f'{a.username}:{a.id}' for a, b in aliens]
            )

        if settings.DRY_RUN:
            logger.info(
                '%d potential kicks in chat %s: %s',
                len(aliens),
                chat.title,
                [a if type(a) == User else a.to_dict() for a, b in aliens],
            )
            return

        if kick_threshold_exceeded(len(aliens), len(members_ids)):
            not_kicked = [a for a, b in aliens]
            too_many = True
            logger.warning(
                '[%s] Kick threshold exceeded for chat %s. Aliens: %s. Members: %s',
                chat.id,
                chat.title,
                len(aliens),
                len(members_ids),
            )
        else:
            for user, db_participant in aliens:
                if await self._try_kick_(user, chat, db_whitelist):
                    await self.user_srv.register_exit(
                        chat_id=chat.id,
                        account_id=db_participant.account_id,
                        action=ACTION.USER_BANNED_BY_BOT,
                    )
                else:
                    not_kicked.append(user)

        if not_kicked:
            logger.info('[%s] %d not kicked users', chat.id, len(not_kicked))
            await self.mail_srv.notify_about_not_kicked_users(
                admins_emails=await self._tu.chat.get_admin_emails(chat.id),
                not_kicked_users=not_kicked,
                chat_title=chat.title,
                chat_id=chat.id,
                too_many=too_many,
            )

        await self._tu.chat.update_last_successful_scan(chat_id=chat.id)
        logger.info('[%s] chat scan complete %s', chat_id, stat)

    async def _merge_participants(
        self, chat: Chat, tg_participants: list[types.ChatMember], db_participants: Iterable[DbParticipant]
    ) -> Iterable[types.User or User]:
        users = [p.user for p in tg_participants]
        tg_ids = [p.id for p in users if p.id is not None]

        for db_participant in db_participants:
            if db_participant.telegram_id in tg_ids:
                continue

            try:
                member = await chat.get_member(user_id=db_participant.telegram_id)
            except BadRequest:
                logger.warning('[%s] participant not found, user %s', chat.id, db_participant)
                # Регистрируем выход для людей которые есть в базе, но нет в чате
                await self.user_srv.register_exit(
                    chat_id=chat.id,
                    account_id=db_participant.account_id,
                    action=ACTION.USER_NOT_PARTICIPANT,
                )
                continue
            except Exception as err:
                logger.exception('[%s] aiogram get_member %s: %s',
                                 chat.id, err.__class__.__name__, err,
                                 exc_info=err)
                continue

            if member.is_chat_member():
                users.append(member.user)
            else:
                # Регистрируем выход для людей которые есть в базе, но нет в чате
                await self.user_srv.register_exit(
                    chat_id=chat.id,
                    account_id=db_participant.account_id,
                    action=ACTION.USER_NOT_PARTICIPANT,
                )
        return users

    async def _register_enter(self, group: Group, participant, is_admin: bool) -> DbParticipant:
        """
        Мы не нашли membership базе, причины могут быть несколько
        - мы не знали что пользователь вошел
        - такого аккаунта вообще нет в базе
        - такой аккаунт есть при синке со стаффом, но мы не знаем его telegram_id
        """
        account_with_user = await self.user_srv.sync_user_account(
            username=participant.username, telegram_id=participant.id, is_bot=participant.is_bot
        )

        account_id = account_with_user.account_id

        membership_id = await self._tu.membership.create(account_id=account_id, group_id=group.id, is_admin=is_admin)

        await self._tu.action.add_action_to_membership(
            action=ACTION.USER_DETECTED_BY_SUBBOT,
            account_id=account_id,
            group_id=group.id,
            membership_id=membership_id,
        )

        db_participant = DbParticipant(
            account_id=account_id,
            username=participant.username,
            telegram_id=participant.id,
            is_bot=participant.is_bot,
            is_active=account_with_user.is_active,
        )

        return db_participant

    @staticmethod
    async def _try_kick_(user, chat: Chat, chat_whitelist) -> bool:
        username = f'{user.id}:"{user.username}"'

        # TASHA-248
        if user.username and (user.username.lower() in settings.TG_WHITELIST):
            logger.error('WHITELIST STATIC user kick attempt %s', username)
            return True

        if is_in_whitelist(user.username, user.id, chat_whitelist):
            logger.error('WHITELIST user kick attempt %s', username)
            return True

        try:
            kicked = await chat.kick(user.id)
        except ChatAdminRequired:
            kicked = False
            logger.warning('[%s] tasha in not admin', chat.id)
        except NotEnoughRightsToRestrict:
            kicked = False
            logger.warning('[%s] NotEnoughRightsToRestrict - %s', chat.id, username)
        except UserIsAnAdministratorOfTheChat:
            kicked = False
            logger.warning('[%s] user is admin - %s', chat.id, username)
        except CantRestrictChatOwner:
            kicked = False
            logger.warning('[%s] user is owner - %s', chat.id, username)
        except BadRequest as err:
            kicked = False
            try:
                await chat.get_member(user_id=user.id)
            except BadRequest:
                kicked = True
            else:
                logger.exception(
                    '[%s][%s] kick participant rise exception %s: %s',
                    chat.id, user.username, err.__class__.__name__, err,
                    exc_info=err
                )
        except Exception as err:
            kicked = False
            logger.error('try kick - %s', user)
            logger.exception(
                '[%s][%s] kick participant rise exception %s: %s',
                chat.id, user.username, err.__class__.__name__, err,
                exc_info=err
            )

        return kicked

    @staticmethod
    async def _is_valid_user(group: Group, db_participant: DbParticipant, participant, db_whitelist) -> bool:
        is_yandex = db_participant.affiliation == 'yandex'
        # Примитивная рль, если у чата опция только Яндекс, то смотрим только сотрудников с affiliation yandex
        has_valid_role = (is_yandex if group.only_yandex else True) or db_participant.is_bot
        is_active_or_bot = db_participant.is_active or db_participant.is_bot
        in_whitelist = is_in_whitelist(participant.username, participant.id, db_whitelist)
        is_valid_user = (is_active_or_bot and has_valid_role) or in_whitelist
        return is_valid_user

    async def _sync_group(self, chat: Chat, group: Group):
        if chat.title != group.title:
            await self._tu.chat.update_title(group.telegram_id, chat.title)
        if chat.type != group.chat_type:
            await self._tu.chat.update_type(group.telegram_id, chat.type)

    async def _sync_user(self, group: Group, participant, is_admin: bool):
        telegram_id = participant.id
        cached_result = SYNC_CACHE.get(telegram_id)
        if cached_result:
            return True

        account_with_user = await self.user_srv.sync_user_account(
            username=participant.username, telegram_id=participant.id, is_bot=participant.is_bot
        )

        mm = await self._tu.membership.get(group.id, account_with_user.account_id)
        if not mm:
            await self._register_enter(
                group=group,
                participant=participant,
                is_admin=is_admin
            )
            return True

        # Посинкать админский статус
        is_current_admin = mm.is_admin
        is_current_active = mm.is_active

        SYNC_CACHE[telegram_id] = mm

        if is_current_active and is_current_admin == is_admin:
            # Активное членство уже есть, ничего не меняем
            return True
        if is_current_admin != is_admin:
            logger.warning('user updated - %s', participant)
            await self._tu.membership.update_admin(mm.id, is_admin)
            await self._tu.action.add_action_to_membership(
                action=ACTION.USER_GRANTED_ADMIN_RIGHTS if is_admin else ACTION.USER_LOST_ADMIN_RIGHTS,
                account_id=account_with_user.account_id,
                group_id=group.id,
                membership_id=mm.id,
            )
        if not mm.is_active:
            await self._tu.membership.update_active(mm.id, True)


def kick_threshold_exceeded(alien_users_count, users_count):
    ratio = alien_users_count / users_count
    if users_count > SMALL_CHAT_SIZE:
        return ratio > KICK_RATIO_THRESHOLD
    else:
        return ratio > SMALL_CHAT_KICK_RATIO_THRESHOLD
