import logging
from dataclasses import dataclass

import yenv
from aiogram.utils.exceptions import BadRequest, MessageIsTooLong
from django.core.mail import EmailMultiAlternatives

from .base import BaseService
from tasha.config import settings
from tasha.lib import cache_storage
from jinja2 import Environment, FileSystemLoader

from ...constants import ACTION

logger = logging.getLogger(__name__)

env = Environment(loader=FileSystemLoader(settings.MAIL_TEMPLATES_PATH))

USER_MESSAGE_SUBJECT = '[@{bot}] Unknown users have been detected in your chat {chat}'
USER_TOO_MANY_MESSAGE_SUBJECT = '[@{bot}] Too many unknown users have been detected in your chat {chat}'
DIALOG_CACHE_TIMEOUT = 24 * 3600


@dataclass
class MailUser:
    username: str = None
    first_name: str = None
    last_name: str = None


def send_mail(to, subject, template_name, context=None):
    logger.info(
        'Mail alert. Template: %s, To: %s, Context: %s',
        template_name, to, context
    )
    if settings.MAIL_ALERT_OFF:
        return 1
    if yenv.type != 'production':
        subject = '[%s] %s' % (yenv.type.upper(), subject)

    if not isinstance(to, list):
        to = [to]
    reply_to = settings.REPLY_TO

    html_template = env.get_template(template_name + '.html')
    text_template = env.get_template(template_name + '.txt')
    html_content = html_template.render(context=context)
    text_content = text_template.render(context=context)

    message = EmailMultiAlternatives(
        subject=subject,
        body=text_content,
        from_email=settings.MAIL_FROM,
        to=to,
        reply_to=reply_to,
        bcc=[settings.BCC_EMAIL],
    )
    message.attach_alternative(html_content, 'text/html')
    message.send()
    # for caching and not alert for a long time
    return 1


def format_user_name(user):
    if isinstance(user, dict):
        user = MailUser(**user)

    print_name = f'{user.first_name or ""} {user.last_name or ""}'.strip()
    username = getattr(user, 'username', '')
    if username:
        print_name = f'{print_name} (@{username})'.strip()

    if getattr(user, 'is_deleted', False) and not print_name:
        print_name = 'DELETED ACCOUNT'

    return print_name


class MailService(BaseService):
    async def notify_about_not_kicked_users(
            self, admins_emails, not_kicked_users, chat_title, chat_id,
            too_many=False, bot_username=settings.TELEGRAM_BOT_USERNAME,
    ):
        if not not_kicked_users:
            logger.error('No users')
            return

        print_users = [format_user_name(user) for user in not_kicked_users]
        context = {
            'bot_username': bot_username,
            'print_users': print_users,
            'group_title': chat_title,
        }

        prev_not_kicked_users = cache_storage.get_key_value(chat_id)
        if set(print_users).issubset(set(prev_not_kicked_users)):
            logger.info('Already notified about chat %s', chat_title)
            return

        if admins_emails:
            subject_template = USER_TOO_MANY_MESSAGE_SUBJECT if too_many else USER_MESSAGE_SUBJECT
            subject = subject_template.format(bot=bot_username, chat=chat_title)

            logger.info('Trying to notify about chat %s', chat_title)
            send_mail(admins_emails, subject, 'telegram_notify', context)
        else:
            message = (
                    'Внимание! В чате имеются пользователи, чьи telegram-аккаунты не являются аккаунтами '
                    'действующих сотрудников Яндекса\n%s' % '\n'.join(print_users)
            )
            logger.info('Trying to send message about %s to chat %s', print_users, chat_title)
            try:
                await self._tu.bot.send_message(chat_id=chat_id, text=message)
            except BadRequest as err:
                logger.exception(
                    '[%s][%s] bot.send_message() %s: %s',
                    chat_id, chat_title, err.__class__.__name__, err,
                    exc_info=err
                )
            except MessageIsTooLong as err:
                logger.exception(
                    '[%s][%s] bot.send_message() %s: %s',
                    chat_id, chat_title, err.__class__.__name__, err,
                    exc_info=err
                )
            else:
                logger.info('Successfully sent message about %s to chat %s', print_users, chat_title)

        cache_storage.set_key_value(chat_id, print_users, DIALOG_CACHE_TIMEOUT)
        await self.register_email(chat_id, not_kicked_users)

    async def register_email(self, chat_id: int, not_kicked_users):
        for user in not_kicked_users:
            logger.info('Adding action EMAIL for chat_telegram_id %s telegram_id %s', chat_id, user.id)
            await self._tu.action.add_action_to_membership(
                action=ACTION.EMAIL,
                telegram_id=user.id,
                chat_telegram_id=chat_id,
            )
            await self._tu.membership.add_notify_date_to_membership(telegram_id=user.id, chat_telegram_id=chat_id)
