import logging
from dataclasses import dataclass
from typing import Optional

from aiogram.utils import exceptions
from aiogram.utils.exceptions import ChatNotFound, BadRequest

from tasha.core.models import User
from tasha.core import TashaUnit
from tasha.core.dbproxy import GroupNotRegistered, TelegramAccountMultipleValue
from tasha.constants import ACTION
from tasha.db.gateways.account import TgAccount
from tasha.db.gateways.user import UserWithAccount

from .mail_service import MailService
from .base import BaseService

UNBAN_BAD_REQUEST_MESSAGE = 'Method is available for supergroup and channel chats only'

logger = logging.getLogger(__name__)


@dataclass
class ChatWithAccount:
    account_id: int
    account_telegram_id: int
    account_telegram_username: str
    group_telegram_id: int


@dataclass
class AccountWithUser(TgAccount):
    account_id: int = None
    is_active: bool = False
    user_id: int = None


class UserService(BaseService):
    def __init__(self, tu: TashaUnit):
        self.mail_service = MailService(tu)
        super().__init__(tu)

    async def get_user(self, username: str) -> User:
        return await self._tu.user.get_by_username(username)

    async def get_chats_with_account(self, user: User) -> [ChatWithAccount]:
        chats_with_acc = await self._tu.user.get_chats_with_acc(user.id)

        return [ChatWithAccount(**chat) for chat in chats_with_acc]

    async def kick_from_chat(self, user: User, chat_id: int, account_id: int, telegram_id: int):
        try:
            chat = await self._tu.bot.get_chat(chat_id=chat_id)
        except exceptions.NotFound:
            logger.warning('[%s][%s] chat not found', user.username, chat_id)
            await self._tu.chat.deactivate(chat_id)
            return
        except Exception as err:
            logger.exception(
                '[%s][%s] Fetching group raised exception %s: %s',
                user.username, chat_id, err.__class__.__name__, err
            )
            return

        member = await chat.get_member(telegram_id)

        if not member.is_chat_member():
            await self.register_exit(
                chat_id=chat.id,
                account_id=account_id,
                action=ACTION.USER_NOT_PARTICIPANT,
            )
            return

        try:
            kicked = await chat.kick(telegram_id)
        except Exception as err:
            kicked = False
            logger.exception(
                '[%s][%s] kick user rise exception %s: %s',
                user.username, chat_id, err.__class__.__name__, err
            )

        if kicked:
            action = ACTION.USER_BANNED_BY_BOT
            await self.register_exit(
                chat_id=chat.id,
                account_id=account_id,
                action=action,
            )
        else:
            logger.warning(
                '[%s][%s] user not kicked',
                user.username, chat_id,
            )
            await self.mail_service.notify_about_not_kicked_users(
                admins_emails=await self._tu.chat.get_admin_emails(chat_id),
                not_kicked_users=[member.user],
                chat_title=chat.title,
                chat_id=chat_id,
                too_many=False,
            )

    async def register_exit(self, chat_id: int, account_id: int, action: str):
        try:
            group = await self._tu.chat.get(chat_id)
        except GroupNotRegistered:
            logger.info('[%s] User account:%s is left from unregistered group', chat_id, account_id)
            return

        await self._tu.chat.register_exit(
            group=group,
            account_id=account_id,
        )
        # Теперь вешаем экшн
        logger.info('[%s] create action %s for account:%s', chat_id, action, account_id)
        await self._tu.action.add_action_to_membership(
            action=action,
            account_id=account_id,
            group_id=group.id,
        )

    async def unban_user(self, user: User):
        chats = await self._tu.user.get_banned_chats(user)

        if not chats:
            logger.info('Not chats to unban found')
            return

        for telegram_id, membership_id, chat in chats:
            try:
                await self._tu.bot.unban_chat_member(
                    chat_id=chat.telegram_id,
                    user_id=telegram_id,
                    only_if_banned=True,
                )
            except ChatNotFound:
                logger.warning(
                    '[%s][%s] tasha lost access to chat. deactivate it',
                    user.username, chat.id
                )
                await self._tu.chat.deactivate(chat.id)
            except BadRequest as err:
                if err.text != UNBAN_BAD_REQUEST_MESSAGE:
                    logger.exception(
                        '[%s][%s] telegram: unban_chat_member exception %s: %s',
                        user.username, chat.id, err.__class__.__name__, err,
                        exc_info=err
                    )
                continue
            except Exception as err:
                logger.exception(
                    '[%s][%s] telegram: unban_chat_member exception %s: %s',
                    user.username, chat.id, err.__class__.__name__, err,
                    exc_info=err
                )

                continue
            else:
                await self._tu.action.add_action_to_membership(
                    ACTION.USER_UNBANNED_BY_BOT,
                    membership_id=membership_id,
                )
                logger.info(
                    '[%s] user %s unbanned in %s',
                    chat.id, user.username, chat.title
                )

    async def update_usernames(
        self, user: User, user_accounts: Optional[list[UserWithAccount]], usernames: set[str]
    ):
        """
        Привязывает указанные телеграм-аккаунты к этому сотруднику
         - Не отвязываем аккаунты Ухуры
        """
        if user_accounts is None:
            user_accounts = []

        new_usernames = {username.lower() for username in usernames}
        old_usernames = {user.telegram_username.lower() for user in user_accounts}
        old_usernames_no_uhura = {user.telegram_username.lower() for user in user_accounts if not user.is_uhura}

        to_link = new_usernames - old_usernames
        # Вот тут вычитаем аккаунты без ухуры, чтоб случайно их не отвязать
        to_unlink = old_usernames_no_uhura - new_usernames

        if to_unlink:
            logger.info('[staff] unlink %s telegrams: %s', user.username, to_unlink)
            await self._tu.user.unlink_telegrams(user, list(to_unlink))

        if to_link and user.is_active:  # К уволенному сотруднику телеграм-аккаунты не привязываем
            logger.info('[staff] link %s telegrams: %s', user.username, to_link)
            for username in to_link:
                await self._tu.user.link_account_to_user(username=username, user_id=user.id)

    async def sync_user_account(self, telegram_id: int, username: str, is_bot: bool):
        """
        Синхронизация телеграмм аккаунта с пользователем
        """
        if username is not None:
            username = username.lower()

        # ищем есть ли в базе такой аккаунт
        row = await self._tu.account.get_account_with_user(telegram_id=telegram_id)
        account_with_user = AccountWithUser(**row) if row else None

        # Мы нашли аккаунт по telegram_id
        if account_with_user:
            saved_username = account_with_user.username
            if username and username != saved_username:
                # Проверим, есть ли у нас в базе пользователь с таким username
                duplicate_account = await self._tu.account.get_by_username(username=username)
                if duplicate_account:
                    # Если мы нашли дублирующий аккаунт с этим пользователем, но без телеграма,
                    # то мерджим эти два аккаунта
                    if duplicate_account.telegram_id is None:
                        await self._tu.account.merge_accounts(telegram_id, username)
                    else:
                        # ToDo: понять как обрабатывать такой случай
                        logger.error(
                            'telegram_id `%s` have username `%s`, but in db this username have telegram_id `%s`',
                            telegram_id, username, duplicate_account.telegram_id
                        )
                        raise TelegramAccountMultipleValue()
                # нет дубликата, просто пользователь поменял username обновляем в базе
                else:
                    await self._tu.account.change_username(telegram_id, username, account_with_user.is_active)
        # Пытаемся найти аккаунт по username
        elif username:
            row = await self._tu.account.get_account_with_user(telegram_id=telegram_id, username=username)
            account_with_user = AccountWithUser(**row) if row else None
            # Если нашли, значит это мы впервые встретили пользователя после синка со стаффом,
            # прописываем ему telegram_id
            if account_with_user:
                await self._tu.account.update_telegram_id(username=username, telegram_id=telegram_id)
                account_with_user.telegram_id = telegram_id

        # Мы не нашли пользователя не по телеграму не по имени, создаем аккаунт
        if account_with_user is None:
            logger.info('username for %s:%s not found', username, telegram_id)
            account = await self._tu.account.create(username, telegram_id, is_bot)
            account_with_user = AccountWithUser(
                account_id=account.id,
                username=account.username,
                telegram_id=account.telegram_id,
                is_bot=account.is_bot,
            )

        if account_with_user.is_bot != is_bot:
            await self._tu.account.update_is_bot(account_with_user.account_id, is_bot)
            account_with_user.is_bot = is_bot

        return account_with_user
