from dataclasses import dataclass

from cachetools import TTLCache

from tasha.config import settings
from tasha.core.dbproxy import DBProxy, GroupNotRegistered


@dataclass
class TgGroup:
    telegram_id: int = None
    title: str = None
    only_yandex: bool = False


class GroupCache(object):
    _cache = None
    db_proxy = None

    def __init__(self, db_proxy: DBProxy, bot_telegram_id: int):
        self._cache = TTLCache(maxsize=100000, ttl=settings.KNOWN_CHATS_BOT_CACHE_TIMEOUT)
        self.db_proxy = db_proxy
        self.bot_telegram_id = bot_telegram_id

    async def invalidate(self):
        chats = await self.db_proxy.get_available_chats(telegram_id=self.bot_telegram_id)
        groups = [TgGroup(**chat) for chat in chats]

        self._cache.clear()
        for group in groups:
            self._cache[group.telegram_id] = group

    async def get_group(self, telegram_id: int) -> TgGroup or None:

        cached_group = self._cache.get(telegram_id)

        if cached_group:
            return cached_group

        try:
            db_group = await self.db_proxy.get_group(chat_id=telegram_id, ignore_deactivated=True)
        except GroupNotRegistered:
            return None

        group = TgGroup(
            telegram_id=db_group['telegram_id'],
            title=db_group['title'],
            only_yandex=db_group['only_yandex']
        )

        self._cache[telegram_id] = group

        return group
