import logging
from typing import Optional

from aiogram import Bot, types
from aiogram.dispatcher.middlewares import BaseMiddleware
from aiogram.types import Chat
from aiogram.utils.exceptions import MigrateToChat, BotKicked

from tasha.config import settings
from tasha.constants import CHAT_TYPE
from tasha.core import dbproxy
from tasha.core.services.user_service import AccountWithUser
from tasha.core.tasha_bot.group_cache import GroupCache, TgGroup
from tasha.core.tasha_bot.lib import utils
from tasha.db.gateways.account import AccountGateway
from tasha.db.gateways.user import UserGateway

logger = logging.getLogger(__name__)


class TashaHandlers(object):
    def __init__(self, bot: Bot):
        self.groups_cache: Optional[GroupCache] = None
        self.bot_data: Optional[types.User] = None
        self.bot = bot
        self.db_proxy = dbproxy.DBProxy()

    async def async_init(self):
        self.bot_data = await self.bot.get_me()
        await self.db_proxy.connect()
        self.groups_cache = GroupCache(db_proxy=self.db_proxy, bot_telegram_id=self.bot_data.id)
        await self.groups_cache.invalidate()
        logger.info('bot info %s', self.bot_data)

    async def handle_new_chat(self, message: types.Message):
        await self.ensure_chat_is_known(message.chat)

        chat = message.chat

        try:
            await self.db_proxy.get_group(chat.id, ignore_deactivated=True)
        except dbproxy.GroupNotRegistered:
            # Эти случаи могут случаться только если до slave'а
            # не долетело создание группы из ensure_chat_is_known()
            logger.info('[%s] joined unknown chat ', chat.id)
        else:
            logger.info('[%s] joined known chat', chat.id)

    async def handle_enter(self, message: types.Message):
        await self.ensure_chat_is_known(message.chat)
        for participant in message.new_chat_members:
            if participant.username.lower() == self.bot_data.username.lower():
                # Skip my self
                continue
            else:
                await self._process_user_enter(message.chat, message.from_user, participant)

    async def handle_exit(self, message: types.Message):
        await self.ensure_chat_is_known(message.chat)

        banner = message.from_user
        chat = message.chat
        gone_user = message.left_chat_member

        banner_id = banner.id
        chat_id = chat.id
        username = gone_user.username
        user_id = gone_user.id
        action = utils.get_exit_action(banner_id, user_id, self.bot.id)
        try:
            await self.db_proxy.register_exit(chat_id, user_id, username, gone_user.is_bot, action)
        except dbproxy.GroupNotRegistered:
            logger.info('[%s] chat not registered', chat_id)

    async def handle_member_update(self, chat_member: types.ChatMemberUpdated):
        await self.ensure_chat_is_known(chat_member.chat)
        chat = chat_member.chat
        user = chat_member.new_chat_member.user

        new_is_admin = chat_member.new_chat_member.status in ('creator', 'administrator')
        await self.db_proxy.update_admin_status(
            chat_id=chat.id,
            telegram_id=user.id,
            is_admin=new_is_admin
        )
        logger.info('[%s] user %s:%s promoted/demoted %s', chat.id, user.username, user.id, new_is_admin)
        if user.id == self.bot_data.id and not new_is_admin:
            logger.error('[%s] Tasha demoted from admin by %s:%s',
                         chat.id, chat_member.from_user.username, chat_member.from_user.id
                         )

    async def handle_join_request(self, chat_member: types.ChatJoinRequest):
        chat = chat_member.chat
        user = chat_member.from_user
        link_owner = chat_member.invite_link.creator

        logger.info(
            '[%s] user %s:%s request to join. from link - %s:%s',
            chat.id, user.username, user.id, link_owner.username, link_owner.id
        )

        await self.ensure_chat_is_known(chat)
        db_chat = await self.db_proxy.get_group(chat.id)

        if not db_chat['auto_approve']:
            return

        # whitelist = await self.db_proxy.get_chat_whitelist_users(chat.id)
        group = await self.groups_cache.get_group(chat.id)
        has_role = await self._check_user_role(group, user.id, user.username, is_bot=user.is_bot)
        is_active_user = await self.db_proxy.is_valid_user(
            user_id=user.id,
            username=user.username,
            is_bot=user.is_bot
        )
        is_valid_user = has_role and is_active_user

        if not is_valid_user:
            logger.info(
                '[%s] user %s:%s not allowed to join. from link - %s:%s',
                chat.id, user.username, user.id, link_owner.username, link_owner.id
            )
            await chat_member.decline()
            return

        await chat_member.approve()
        await self._process_user_enter(chat, link_owner, user)

    async def ensure_chat_is_known(self, chat: types.Chat):
        """
        Убедиться что мы знаем о чате и добавить в базу чат в случае если не знали до сих пор
        """

        if chat is None or chat.type == CHAT_TYPE.PRIVATE:
            return

        new_chat_id = chat.id
        # // check cache
        group = await self.groups_cache.get_group(new_chat_id)
        if group:
            return

        await self._register_chat(new_chat_id)

    async def _register_chat(self, chat_id: int):
        try:
            chat_details = await self.bot.get_chat(chat_id=chat_id)
            bot_membership = await self.bot.get_chat_member(chat_id=chat_id, user_id=self.bot_data.id)
        except Exception as err:
            logger.exception('[%s] fetch chat error %s: %s', chat_id, err.__class__.__name__, err)
            return

        logger.info(
            '[%s] %s detected %s:%s',
            chat_id, chat_details.type, chat_id, chat_details.title
        )

        # Members count could give some errors if no admin, migrated etc
        try:
            members_count = await self.bot.get_chat_member_count(chat_id=chat_id)
        except MigrateToChat as err:
            return await self._register_chat(err.migrate_to_chat_id)
        except BotKicked:
            # Tasha not admin
            members_count = 0

        god_bot_number = utils.get_shard_number()

        await self.db_proxy.register_new_chat(
            chat_telegram_id=chat_id,
            chat_title=chat_details.title,
            chat_members_count=members_count,
            chat_type=chat_details.type,
            bot_telegram_id=self.bot_data.id,
            bot_is_admin=bot_membership['status'] == 'administrator',
            god_bot_number=god_bot_number,
        )

    async def _process_user_enter(self, chat: types.Chat, inviter: types.User, participant: types.User):
        chat_id = chat.id
        username = participant.username
        telegram_id = participant.id

        logger.info('[%s] new_chat_member: %s', chat_id, username)

        whitelist = await self.db_proxy.get_chat_whitelist_users(chat_id)

        # TASHA-248
        if username and (username.lower() in settings.TG_WHITELIST):
            logger.warning('[%s] WHITELIST STATIC user enter %s', chat_id, username)
            return

        action = utils.get_enter_action(inviter.id, telegram_id)
        group = await self.groups_cache.get_group(chat_id)
        if not group:
            logger.error('[%s] group is not registered', chat_id)

        try:
            membership_id, is_active_user = await self.db_proxy.register_enter(
                chat_id, telegram_id, username, participant.is_bot, action
            )

            has_role = await self._check_user_role(group, telegram_id, username, is_bot=participant.is_bot)
            is_active_user = is_active_user and has_role

            if not utils.is_valid_user(chat_id, participant, is_active_user, whitelist):
                logger.info('[%s] User "%s":%s not valid. Try kick', chat_id, username, telegram_id)
                try:
                    result = await chat.kick(telegram_id)
                except Exception as exc:
                    result = None
                    logger.exception(
                        '[%s] Kick user %s raised exception %s: %s',
                        chat_id, telegram_id, exc.__class__.__name__, exc
                    )

                if not result:
                    logger.error('[%s] Kick failed "%s":%s', chat_id, username, telegram_id)
                    await utils.notify_admin(self.db_proxy, chat, participant)

        except dbproxy.GroupNotRegistered:
            logger.info('[%s] Group is not registered', chat_id)
        except dbproxy.UserMultipleValue:
            logger.info(
                '[%s] username `%s` have telegram_id %s, but they have different users',
                chat_id, username, telegram_id
            )

    async def _check_user_role(self, group: TgGroup, telegram_id: int, username: str, is_bot: bool):
        if not group.only_yandex or is_bot:
            return True
        async with self.db_proxy.acquire_connection() as conn:
            account_gw = AccountGateway(conn)
            user_gw = UserGateway(conn)
            row = await account_gw.get_account_with_user(telegram_id=telegram_id, username=username)
            account_with_user = AccountWithUser(**row) if row else None
            user = await user_gw.get(account_with_user.user_id)
            affiliation = user.affiliation
            return affiliation == 'yandex'


chat_events_holders = [
    'chat_join_request', 'message', 'edited_message',
    'channel_post', 'edited_channel_post', 'my_chat_member', 'chat_member'
]


class EnsureGroupMiddleware(BaseMiddleware):
    def __init__(self, bot_handlers: TashaHandlers):
        self.bot_handlers = bot_handlers
        super(EnsureGroupMiddleware, self).__init__()

    async def on_pre_process_update(self, update: types.Update, *args):
        chat: Optional[Chat] = None
        for holder in chat_events_holders:
            if update[holder]:
                chat = update[holder].chat
                break

        if not chat:
            return
        try:
            await self.bot_handlers.ensure_chat_is_known(chat)
        except Exception as exc:
            logger.exception(
                '[%s] ensure chat exception %s: %s',
                chat.id, exc.__class__.__name__, exc
            )
