import uuid
from typing import Any

from aiogram import Bot
from arq import ArqRedis
from asyncpg import Connection

from tasha.config import settings
from tasha.core.dbproxy import DBProxy
from tasha.db.gateways.account import AccountGateway
from tasha.db.gateways.action import ActionGateway
from tasha.db.gateways.chat import ChatGateway
from tasha.db.gateways.membership import MembershipGateway
from tasha.db.gateways.sync_time import SyncTimeGateway
from tasha.db.gateways.user import UserGateway
from tasha.db.gateways.whitelist import WhitelistGateway
from tasha.lib.utils import make_hash


class TashaUnit:
    """
    TBD: описание

    Пример использования:
    uow = TashaUnit(conn=conn, redis=redis)
    async with uow:
        await uow.trips.update(trip_id=trip_id, **fields)
        uow.add_job('trip_updated', trip_id=trip_id)

    """

    def __init__(self, conn: Connection, db_proxy: DBProxy, bot: Bot, redis: ArqRedis = None):
        self._conn = conn
        self._db_proxy = db_proxy
        self._redis = redis

        self.sync_time: SyncTimeGateway = SyncTimeGateway(conn)
        self.user: UserGateway = UserGateway(conn)
        self.chat: ChatGateway = ChatGateway(conn)
        self.action: ActionGateway = ActionGateway(conn)
        self.account: AccountGateway = AccountGateway(conn)
        self.membership: MembershipGateway = MembershipGateway(conn)
        self.whitelist: WhitelistGateway = WhitelistGateway(conn)

        self.bot: Bot = bot

        self._jobs: list[tuple[str, dict]] = []

    async def __aenter__(self):
        return self

    async def __aexit__(self, exn_type, exn_value, traceback):
        if exn_type is None:
            await self._enqueue_jobs()

    def add_job(self, job_name: str, **kwargs: Any) -> None:
        """
        Добавляет таску, чтобы ее выполнить после коммита транзакции

        :param job_name: задача, которую нужно запланировать
        :param kwargs: аргументы, которые нужно передать в задачу
        """
        assert self._redis is not None
        self._jobs.append((job_name, kwargs))

    def add_slow_job(self, job_name: str, **kwargs: Any) -> None:
        kwargs['_queue_name'] = settings.ARQ_SLOW_QUEUE_NAME
        self.add_job(job_name=job_name, **kwargs)

    async def run_job(
        self,
        job_name: str,
        unique: bool = True,
        **kwargs: Any,
    ) -> None:
        """
        Запускает фоновую задачу

        :param job_name: задача, которую нужно запустить
        :param unique: блокировать запуск задачи с таким же набором аргументов
        :param kwargs: аргументы, которые нужно передать в задачу
        """
        assert self._redis is not None
        job_id = kwargs.get('_job_id', self._get_job_id(unique, **kwargs))
        kwargs['_job_id'] = f'{job_name}_{job_id}'
        await self._redis.enqueue_job(job_name, **kwargs)

    @property
    def redis(self) -> ArqRedis:
        return self._redis

    async def _enqueue_jobs(self) -> None:
        for job_name, kwargs in self._jobs:
            await self.run_job(job_name, **kwargs)
        self._jobs = []

    @staticmethod
    def _is_of_simple_type(value: Any) -> bool:
        """
        Проверяет, является ли значение целым числом, строкой
        либо списком из 1 элемента
        """
        return (
            isinstance(value, (int, str))
            or (
                isinstance(value, list)
                and len(value) == 1
                and isinstance(value[0], (int, str))
            )
        )

    def _build_job_id_from_args(self, **job_kwargs: Any) -> str:
        simple_keys = [
            key
            for key in sorted(job_kwargs.keys())
            if self._is_of_simple_type(job_kwargs[key])
        ]
        values = [str(job_kwargs[key]) for key in simple_keys]
        hashed_other_kwargs = make_hash({
            key: value
            for key, value in job_kwargs.items()
            if key not in simple_keys
        })
        values.append(hashed_other_kwargs)
        return '_'.join(values)

    def _get_job_id(self, unique: bool, **job_kwargs: Any) -> str:
        if unique:
            return self._build_job_id_from_args(**job_kwargs)
        return uuid.uuid4().hex
