import inspect
import logging
from functools import partial

from aiogram import Bot
from arq import cron

from tasha.config import settings
from tasha.core.dbproxy import DBProxy
from tasha.external.telegram import TelegramClient

from .redis_settings import get_redis_settings, create_redis_pool

logger = logging.getLogger(__name__)


async def startup(ctx):
    db_proxy = DBProxy()
    await db_proxy.connect()
    ctx['db'] = db_proxy
    ctx['bot'] = Bot(token=settings.TELEGRAM_TOKEN)
    Bot.set_current(ctx['bot'])
    ctx['redis'] = await create_redis_pool()
    client = TelegramClient()
    await client.start()
    ctx['client'] = client

    logger.info('Worker inited')


async def shutdown(ctx):
    await ctx['db'].close()
    bot: Bot = ctx['bot']
    session = await bot.get_session()
    await session.close()

    await ctx['client'].stop()

    logger.info('Worker shutdown')


def is_job(is_cron: bool, task):
    if not inspect.isfunction(task):
        return False
    arq_settings = getattr(task, 'arq_config', None)
    if arq_settings is None:
        return False

    settings_is_cron = arq_settings['is_cron'] or False

    if is_cron != settings_is_cron:
        return False

    return True


def get_jobs(tasks, is_cron: bool = False):
    validate_job = partial(is_job, is_cron)
    jobs_from_tasks = inspect.getmembers(tasks, validate_job)
    return [function for name, function in jobs_from_tasks]


def get_cron_jobs(tasks):
    jobs = get_jobs(tasks, is_cron=True)
    return [cron(job, **job.arq_config['cron_kwargs']) for job in jobs]


class BaseWorkerSettings:
    on_startup = startup
    on_shutdown = shutdown
    redis_settings = get_redis_settings()
    functions = []
    cron_jobs = []
    max_tries = settings.ARQ_MAX_JOB_TRIES
    max_jobs = settings.ARQ_MAX_JOB_COUNT
