import functools
import logging
from functools import wraps

from arq import Retry
from django.conf import settings
from sentry_sdk import capture_exception, add_breadcrumb, push_scope

from tasha.lib.log_context import arq_log_context

logger = logging.getLogger(__name__)


async def _execute_job(job_coro, ctx, *args, **kwargs):
    with push_scope() as scope:
        scope.set_tag('source', job_coro.__qualname__)

        try:
            return await job_coro(ctx, *args, **kwargs)
        except Retry as e:
            capture_exception(e.__cause__)
            raise
        except Exception as e:
            capture_exception(e)
            raise


def job(is_fast=True):
    def decorator(func):
        @wraps(func)
        async def wrapper(ctx, *args, **kwargs):
            ctx['is_last_try'] = ctx['job_try'] == settings.ARQ_MAX_JOB_TRIES
            with arq_log_context(func, ctx):
                result = await _execute_job(func, ctx, *args, **kwargs)

            return result

        wrapper.arq_config = {
            'is_cron': False,
            'is_fast': is_fast,
        }
        return wrapper
    return decorator


def cron_job(predicate=True, is_fast=True, **cron_kwargs):
    def decorator(func):
        @wraps(func)
        async def wrapper(ctx, *args, **kwargs):
            ctx['is_last_try'] = ctx['job_try'] == settings.ARQ_MAX_JOB_TRIES
            with arq_log_context(func, ctx):
                result = await _execute_job(func, ctx, *args, **kwargs)

            return result

        wrapper.arq_config = {
            'is_cron': True,
            'predicate': predicate,
            'cron_kwargs': cron_kwargs,
            'is_fast': is_fast,
        }
        return wrapper

    return decorator


def default_retry_defer_callable(job_try):
    return 30 * job_try


def retry_on_exception(retry_defer_callable=default_retry_defer_callable):
    def decorator(coro):
        @functools.wraps(coro)
        async def wrapped(ctx, *args, **kwargs):
            try:
                return await coro(ctx, *args, **kwargs)
            except Exception as err:
                capture_exception(err)
                logger.exception(
                    'Exception, msg: %s, in task %s:%s on %s try',
                    err,
                    ctx['job_id'],
                    coro.__qualname__,
                    ctx['job_try'],
                )
                raise Retry(defer=retry_defer_callable(ctx['job_try'])) from err

        return wrapped

    return decorator
