import logging
from dataclasses import dataclass

from tasha.constants import ACTION
from tasha.core.dbproxy import UserMultipleValue
from tasha.core.models.account import Account
from tasha.db.gateways.base import DBGateway

logger = logging.getLogger(__name__)


@dataclass
class TgAccount:
    telegram_id: int = None
    username: str = None
    is_bot: bool = False


class AccountGateway(DBGateway):
    async def create(self, username: str, telegram_id: int, is_bot: bool):
        insert_query = '''
            INSERT INTO tasha_telegramaccount
                (user_id, username, telegram_id, is_bot)
            VALUES
                (null, $1, $2::numeric, $3)
            RETURNING *;
            '''
        row = await self.conn.fetchrow(insert_query, username, telegram_id, is_bot)
        return Account(**row) if row else None

    async def get_by_username(self, username: str):
        select_query = '''
            SELECT *
              FROM tasha_telegramaccount
             WHERE username = $1
        '''
        row = await self.conn.fetchrow(select_query, username)
        return Account(**row) if row else None

    async def update_telegram_id(self, username: str, telegram_id: int):
        await self.conn.execute(
            '''
            UPDATE tasha_telegramaccount
            SET telegram_id = $2
            WHERE username = $1
            ''', username, telegram_id
        )

    async def update_is_bot(self, account_id: int, is_bot: bool):
        await self.conn.execute(
            '''
            UPDATE tasha_telegramaccount
            SET is_bot = $1
            WHERE id = $2
            ''', is_bot, account_id
        )

    async def get_account_with_user(self, telegram_id: int, username: str = None):
        """
        Возвращает найденный аккаунт по telegram_id или username c привязкой к user_id
        """
        request = '''
            SELECT tasha_user.id as user_id,
                   tasha_user.is_active as is_active,
                   tasha_telegramaccount.telegram_id as telegram_id,
                   tasha_telegramaccount.is_bot as is_bot,
                   tasha_telegramaccount.id as account_id,
                   tasha_telegramaccount.username as username
              FROM tasha_telegramaccount LEFT OUTER JOIN tasha_user
                ON tasha_telegramaccount.user_id = tasha_user.id
             WHERE tasha_telegramaccount.telegram_id = $1::numeric
        '''
        arguments = [telegram_id]
        if username:
            request += ' OR tasha_telegramaccount.username = $2'
            arguments.append(username)
        return await self.conn.fetchrow(request, *arguments)

    async def merge_accounts(self, telegram_id: int, username: str):
        # Проверим, что для текущего username нет чатов
        ASSERT_MEMBERSHIP_QUERY = '''
            SELECT COUNT(*) as count
              FROM tasha_membership
              JOIN tasha_telegramaccount
                ON tasha_membership.account_id = tasha_telegramaccount.id
             WHERE tasha_telegramaccount.username = $1
        '''

        SELECT_USER_FROM_USERNAME_QUERY = '''
            SELECT user_id
              FROM tasha_telegramaccount
             WHERE username = $1
        '''

        SELECT_USER_FROM_TELEGRAM_ID_QUERY = '''
            SELECT user_id
              FROM tasha_telegramaccount
             WHERE telegram_id = $1
        '''

        DELETE_QUERY = '''
            DELETE FROM tasha_telegramaccount
             WHERE username = $1;
        '''

        UPDATE_QUERY = '''
            UPDATE tasha_telegramaccount
               SET username = $1, user_id = $2
             WHERE telegram_id = $3;
        '''

        count = (await self.conn.fetch(ASSERT_MEMBERSHIP_QUERY, username))[0]['count']
        assert count == 0, f'username `{username}` with no telegram_id have {count} memberships'
        # Получим пользователя, привязанного к username
        username_user = (await self.conn.fetch(SELECT_USER_FROM_USERNAME_QUERY, username))[0]['user_id']
        telegram_id_user = (await self.conn.fetch(SELECT_USER_FROM_TELEGRAM_ID_QUERY, telegram_id))[0]['user_id']
        if username_user is not None and telegram_id_user is not None and username_user != telegram_id_user:
            logger.error(
                'username `%s` have telegram_id %s, but they have different users (%s, %s)',
                username, telegram_id, username_user, telegram_id_user
            )
            raise UserMultipleValue()

        # Удалим username и обновим данные
        user_id = username_user or telegram_id_user
        async with self.conn.transaction():
            await self.conn.execute(DELETE_QUERY, username)
            await self.conn.execute(UPDATE_QUERY, username, user_id, telegram_id)

    async def change_username(self, telegram_id: int, username: str, user_is_active: bool):
        await self.conn.execute(
            '''
            UPDATE tasha_telegramaccount
            SET username = $2
            WHERE telegram_id = $1
            ''', telegram_id, username,
        )
        if user_is_active:
            add_renaming_actions_sql = f'''
                WITH action_values AS(
                    SELECT
                        CURRENT_TIMESTAMP, '{ACTION.USER_CHANGED_USERNAME}', tasha_membership.id
                    FROM tasha_membership
                    JOIN tasha_telegramaccount ON tasha_membership.account_id = tasha_telegramaccount.id
                    WHERE
                        tasha_membership.is_active = TRUE AND tasha_telegramaccount.telegram_id = $1
                )
                INSERT INTO tasha_actions (added, action, membership_id)
                SELECT * FROM action_values
            '''
            await self.conn.execute(add_renaming_actions_sql, telegram_id)
