import logging
from dataclasses import dataclass
from datetime import timezone
from typing import Iterable, List

from tasha.config import settings
from tasha.constants import ACTION
from tasha.core.dbproxy import GroupNotRegistered, UserMultipleValue
from tasha.core.models import User
from tasha.core.models.group import Group
from tasha.db.gateways.base import DBGateway

logger = logging.getLogger(__name__)


@dataclass
class DbParticipant:
    account_id: int = None
    username: str = None
    telegram_id: int = None
    is_bot: bool = False
    is_active: bool = False
    affiliation: str = None


class ChatGateway(DBGateway):
    async def get(self, chat_id, ignore_deactivated=False):
        query = '''
            SELECT *
            FROM tasha_tggroupinfo
            WHERE telegram_id = $1::numeric
        '''
        if ignore_deactivated:
            query += ' AND NOT deactivated'

        row = await self.conn.fetchrow(query, chat_id)

        if not row:
            raise GroupNotRegistered('Group with telegram_id %s not found' % chat_id)

        return Group(**row)

    async def deactivate(self, chat_id: int):
        await self.conn.execute('''
            UPDATE tasha_tggroupinfo
            SET deactivated = TRUE
            WHERE telegram_id = $1
        ''', chat_id)

    async def register_exit(self, group: Group, account_id: int):
        """
        Деактивирует соответствующий мембершип
        """
        if settings.DRY_RUN:
            return

        await self.conn.execute('''
              INSERT INTO tasha_membership (account_id, group_id, is_active)
              VALUES ($1::numeric, $2::numeric, false)
              ON CONFLICT (account_id, group_id)
              DO UPDATE SET is_active = false
        ''', account_id, group.id)

    async def get_admin_emails(self, chat_id: int) -> List[str]:
        SELECT_ADMINS_EMAILS_QUERY = '''
            SELECT tasha_user.email
            FROM tasha_membership
                JOIN tasha_tggroupinfo ON (tasha_membership.group_id = tasha_tggroupinfo.id)
                JOIN tasha_telegramaccount ON (tasha_membership.account_id = tasha_telegramaccount.id)
                JOIN tasha_user ON (tasha_telegramaccount.user_id = tasha_user.id)
            WHERE
                tasha_tggroupinfo.telegram_id = $1
                AND tasha_membership.is_active = TRUE
                AND tasha_tggroupinfo.deactivated = FALSE
                AND tasha_user.is_active = TRUE
                AND tasha_user.email IS NOT NULL
                AND tasha_membership.is_admin = TRUE
                AND NOT tasha_user.username = 'robot-uhura'
        '''
        rows = await self.conn.fetch(SELECT_ADMINS_EMAILS_QUERY, chat_id)
        return [x['email'] for x in rows]

    async def get_least_scanned_chats(self, limit: int) -> Iterable[int]:
        select_query = '''
            SELECT DISTINCT telegram_id, last_successful_scan
            FROM
                tasha_tggroupinfo
            WHERE
                NOT tasha_tggroupinfo.deactivated
            ORDER BY last_successful_scan NULLS FIRST
            LIMIT $1
                '''

        rows = await self.conn.fetch(select_query, limit)

        return [row['telegram_id'] for row in rows]

    async def get_participants(self, chat_id: int) -> Iterable[DbParticipant]:
        select_query = '''
            SELECT
                ta.username, ta.telegram_id, ta.is_bot, ta.id as account_id,
                tu.is_active, tu.affiliation as affiliation
            FROM
                tasha_membership as tm
                INNER JOIN tasha_tggroupinfo tg ON (tm.group_id = tg.id)
                INNER JOIN tasha_telegramaccount ta ON (tm.account_id = ta.id)
                LEFT OUTER JOIN tasha_user tu ON (ta.user_id = tu.id)
            WHERE
                tg.telegram_id = $1
                AND tm.is_active
                AND NOT tg.deactivated
                '''
        rows = await self.conn.fetch(select_query, chat_id)
        return [DbParticipant(**row) for row in rows]

    async def update_last_successful_scan(self, chat_id: int):
        update_query = '''
            UPDATE tasha_tggroupinfo
            SET last_successful_scan = CURRENT_TIMESTAMP
            WHERE telegram_id = $1
        '''
        await self.conn.execute(update_query, chat_id)

    async def get_members_count(self, chat_id: int):
        select_query = '''
                    SELECT count(*) as count
                    FROM
                        tasha_membership as tm
                        INNER JOIN tasha_tggroupinfo tg ON (tm.group_id = tg.id)
                    WHERE
                        tg.telegram_id = $1
                        AND tm.is_active
                        AND NOT tg.deactivated
                        '''
        row = await self.conn.fetchrow(select_query, chat_id)
        return row['count'] if row else None

    async def update_title(self, chat_id, new_title: str):
        update_query = '''
                    UPDATE tasha_tggroupinfo
                    SET title = $2
                    WHERE telegram_id = $1
                '''
        await self.conn.execute(update_query, chat_id, new_title)

    async def update_type(self, chat_id: int, new_type: str):
        update_query = '''
                    UPDATE tasha_tggroupinfo
                    SET chat_type = $2
                    WHERE telegram_id = $1
                '''
        await self.conn.execute(update_query, chat_id, new_type)
