import logging

from tasha.config import settings
from tasha.core.models.membership import Membership
from tasha.db.gateways.base import DBGateway

logger = logging.getLogger(__name__)


class MembershipGateway(DBGateway):
    async def create(self, account_id: int, group_id: int, is_admin: bool) -> int:
        insert_query = '''
          INSERT INTO tasha_membership (account_id, group_id, is_active, is_admin)
          VALUES ($1::numeric, $2::numeric, true, $3)
          ON CONFLICT (account_id, group_id)
          DO UPDATE SET is_active = TRUE, is_admin = $3, first_notified_from_email = NULL
          RETURNING id
        '''
        row = await self.conn.fetchrow(insert_query, account_id, group_id, is_admin)
        return row['id']

    async def add_notify_date_to_membership(self, telegram_id, chat_telegram_id):
        SELECT_QUERY = '''
            SELECT tasha_membership.id, tasha_membership.first_notified_from_email
              FROM tasha_membership
              JOIN tasha_telegramaccount ON tasha_membership.account_id = tasha_telegramaccount.id
              JOIN tasha_tggroupinfo ON tasha_membership.group_id = tasha_tggroupinfo.id
             WHERE tasha_telegramaccount.telegram_id = $1 AND
                   tasha_tggroupinfo.telegram_id = $2
        '''

        UPDATE_QUERY = '''
            UPDATE tasha_membership SET first_notified_from_email = CURRENT_TIMESTAMP
             WHERE tasha_membership.id = $1
        '''

        pk, current_value = (await self.conn.fetch(SELECT_QUERY, telegram_id, chat_telegram_id))[0]
        if current_value is None:
            await self.conn.execute(UPDATE_QUERY, pk)

    async def get(self, group_id: int, account_id: int):
        select_query = '''
            SELECT *
            FROM tasha_membership as mm
            WHERE mm.account_id = $1
                AND mm.group_id = $2
        '''

        row = await self.conn.fetchrow(select_query, account_id, group_id)
        return Membership(**row) if row else None

    async def update_admin(self, mm_id: int, is_admin: bool):
        update_query = '''
            UPDATE tasha_membership
            SET is_admin = $2
            WHERE tasha_membership.id = $1
        '''
        await self.conn.execute(update_query, mm_id, is_admin)

    async def update_active(self, mm_id: int, is_active: bool):
        update_query = '''
                    UPDATE tasha_membership
                    SET is_active = $2
                    WHERE tasha_membership.id = $1
                '''
        await self.conn.execute(update_query, mm_id, is_active)
