import logging
from dataclasses import dataclass
from datetime import timezone, datetime
from typing import Iterable

from tasha.config import settings
from tasha.constants import ACTION
from tasha.core.models import User
from tasha.core.models.group import Group
from tasha.db.gateways.base import DBGateway

logger = logging.getLogger(__name__)

fields_w_order = ['username', 'email', 'is_active', 'quit_at', 'leave_at', 'join_at', 'organization_id', 'affiliation']


@dataclass
class UserWithAccount:
    username: str
    telegram_username: str
    telegram_id: int
    is_uhura: bool


class UserGateway(DBGateway):
    async def get(self, user_id: int):
        select_query = '''
            SELECT *
            FROM tasha_user as usr
            WHERE usr.id = $1
                    '''
        row = await self.conn.fetchrow(select_query, user_id)
        return User(**row) if row else None

    async def create(self, **fields):
        insert_query = '''
            INSERT INTO tasha_user
            (username, email, is_active, quit_at, join_at, organization_id, affiliation)
            VALUES ($1, $2, $3::bool, $4::date, $5::date, $6::int, $7)
            ON CONFLICT (username)
            DO UPDATE SET join_at = $5, organization_id = $6, is_active = true
            RETURNING *
                '''
        row = await self.conn.fetchrow(insert_query, *[fields[field] for field in [
            'username', 'email', 'is_active', 'leave_at', 'join_at', 'organization_id', 'affiliation'
        ]])
        return User(**row)

    async def update(self, user_id: int, **fields):
        update_query = '''
            UPDATE tasha_user
            SET
                username = $2, email = $3, is_active=$4,
                quit_at = $5::date, leave_at = $6, join_at = $7::date, organization_id = $8,
                affiliation = $9
            WHERE id = $1
                '''

        await self.conn.execute(update_query, user_id, *[fields[field] for field in fields_w_order])

    async def get_users_by_usernames(self, usernames: Iterable[str]):
        select_query = '''
            SELECT *
            FROM tasha_user as usr
            WHERE
                usr.username = ANY($1::text[])
                       '''
        rows = await self.conn.fetch(select_query, usernames)
        return [User(**row) for row in rows]

    async def get_users_accounts(self, usernames: Iterable[str]) -> [UserWithAccount]:
        """
        Возвращает аккаунты пользователя
        - отмечаем аккаунты от Ухуры, для того чтоб не отвзязывать их
        """
        select_query = '''
            SELECT
                tu.username as username,
                ta.username as telegram_username,
                ta.telegram_id as telegram_id,
                ta.created_with_uhura as is_uhura
            FROM tasha_telegramaccount as ta
            LEFT JOIN tasha_user tu on tu.id = ta.user_id
            WHERE
                tu.username = ANY($1::text[])
                AND ta.username NOTNULL
                       '''
        rows = await self.conn.fetch(select_query, usernames)
        return [UserWithAccount(**row) for row in rows]

    async def get_by_username(self, username: str):
        select_query = '''
            SELECT *
            FROM tasha_user as usr
            WHERE usr.username = $1
                       '''
        row = await self.conn.fetchrow(select_query, username)
        return User(**row) if row else None

    async def get_chats_with_acc(self, user_id: int):
        """
        Получаем все участия пользователя (acccount-group)
        В том числе с участием аккаунтов ухуры
        """
        select_query = '''
                    SELECT
                        acc.id as account_id,
                        acc.telegram_id as account_telegram_id,
                        acc.username as account_telegram_username,
                        tg.telegram_id as group_telegram_id
                    FROM tasha_telegramaccount as acc
                    INNER JOIN tasha_user tu on tu.id = acc.user_id
                    INNER JOIN tasha_membership tm on acc.id = tm.account_id
                    INNER JOIN tasha_tggroupinfo tg on tg.id = tm.group_id
                    WHERE
                        tu.id = $1
                        AND NOT tg.deactivated
                        AND tm.is_active
                '''
        rows = await self.conn.fetch(select_query, user_id)
        return rows

    async def unlink_telegrams(self, user: User, usernames_to_delete: Iterable[str]):
        """
        Отвязывает телеграм usernames от пользователя
        """
        select_account_ids_query = '''
            SELECT DISTINCT acc.id as account_id
            FROM tasha_telegramaccount as acc
            WHERE
                acc.user_id = $1::numeric
                AND acc.username = ANY($2::text[])
        '''

        select_memberships_query = '''
            SELECT DISTINCT mm.id as mm_id
            FROM tasha_membership as mm
            WHERE
                mm.account_id = ANY($1::numeric[])
            AND mm.is_active
        '''

        unlink_accounts_query = '''
            UPDATE tasha_telegramaccount
            SET user_id = NULL
            WHERE id = ANY($1::numeric[])
        '''

        create_action_query = '''
            INSERT INTO tasha_actions
            (added, action, membership_id)
            VALUES ($1::date, $2, $3::numeric)
            RETURNING id
        '''

        account_rows = await self.conn.fetch(select_account_ids_query, user.id, usernames_to_delete)
        user_account_ids = [row['account_id'] for row in account_rows]

        mm_rows = await self.conn.fetch(select_memberships_query, user_account_ids)
        memberships_ids = [row['mm_id'] for row in mm_rows]

        date = datetime.now()

        async with self.conn.transaction():
            await self.conn.execute(unlink_accounts_query, user_account_ids)
            await self.conn.executemany(create_action_query, [
                (date, ACTION.USER_DELETED_TELEGRAM, mm_id) for mm_id in memberships_ids
            ])

    async def get_account_usernames(self, user_id: int):
        select_query = '''
            SELECT DISTINCT ta.username as username
            FROM tasha_telegramaccount as ta
            WHERE
                ta.user_id = $1::numeric
            AND ta.username IS NOT NULL
                       '''
        rows = await self.conn.fetch(select_query, user_id)
        return [row['username'] for row in rows]

    async def link_account_to_user(
        self,
        username: str, user_id: int, is_bot: bool = False
    ):
        """
        Привязываем аккаунт к пользователю
        - если такого нет создаем
        - если такой есть но у него не прописан пользователь то прописываем пользователя
        - в ином случае падаем в конфликт
        """
        insert_query = '''
            INSERT INTO tasha_telegramaccount
            (username, user_id, is_bot)
            VALUES ($1, $2, $3)
            ON CONFLICT (username) WHERE user_id IS NULL
            DO UPDATE SET user_id = $2
            RETURNING id
        '''
        row = await self.conn.fetchrow(insert_query, username, user_id, is_bot)
        return row

    async def get_banned_chats(self, user: User) -> list[tuple[int, int, Group]]:
        select_query = '''
            SELECT
                tt2.id, tt2.deactivated, tt2.last_successful_scan, tt2.title,
                tt2.telegram_id, tt2.auto_approve, tt2.chat_type,
                ta.telegram_id as user_telegram_id,
                m.id as membership_id
            FROM (
                    SELECT act.membership_id, MAX(act.added) as added
                    FROM tasha_actions act
                    JOIN tasha_membership tm on tm.id = act.membership_id
                    JOIN tasha_telegramaccount ta on ta.id = tm.account_id
                    JOIN tasha_tggroupinfo t on t.id = tm.group_id
                    WHERE
                        ta.user_id = $1
                    AND NOT t.deactivated
                    GROUP BY act.membership_id
                 ) as latest_actions
            INNER JOIN tasha_actions actions USING (membership_id, added)
            INNER JOIN tasha_membership m on m.id = actions.membership_id
            INNER JOIN tasha_tggroupinfo tt2 on tt2.id = m.group_id
            INNER JOIN tasha_telegramaccount ta on ta.id = m.account_id
            WHERE
                actions.action = ANY($2::text[])
        '''
        action_filter = ACTION.user_kicked - {ACTION.USER_BANNED_BY_USER}
        rows = await self.conn.fetch(select_query, user.id, action_filter)
        return [(row['user_telegram_id'], row['membership_id'], Group(**row)) for row in rows]
