from typing import Iterable, Iterator

from slackclient import SlackClient


class TashaSlackAPIException(Exception):
    pass


class TashaSlackAPI:

    token: str
    page_size: int
    _client: SlackClient = None

    def __init__(self, token: str, page_size: int):
        self.token = token
        self.page_size = page_size

    @property
    def client(self) -> SlackClient:
        if self._client is None:
            self._client = SlackClient(token=self.token)
        return self._client

    def _call(self, method: str, **kwargs) -> dict:
        slack_result = self.client.api_call(method=method, **kwargs)
        if not slack_result['ok']:
            raise TashaSlackAPIException('Error while executing `{}`: {}'.format(method, slack_result['error']))
        return slack_result

    def _call_paginated(self, method: str, **kwargs) -> Iterator[dict]:
        first_page = self._call(method=method, limit=self.page_size, **kwargs)
        yield first_page
        next_cursor = first_page['response_metadata']['next_cursor']
        while next_cursor != '':
            slack_result = self._call(method=method, cursor=next_cursor, limit=self.page_size, **kwargs)
            next_cursor = slack_result['response_metadata']['next_cursor']
            yield slack_result

    def get_users_id_login_map(self, team_id: str) -> dict[str, str]:
        result = {}
        for user_page in self._call_paginated(method='users.list', team_id=team_id):
            for user in user_page['members']:
                result[user['id']] = user['name']
        return result

    def get_channels(self, team_id: str, types: Iterable[str], exclude_archived: bool = False) -> Iterator[dict]:
        channel_pages = self._call_paginated(
            method='conversations.list',
            team_id=team_id,
            types=','.join(types),
            exclude_archived=exclude_archived,
        )
        for channel_page in channel_pages:
            yield from channel_page['channels']

    def get_channel_members(self, channel_id: str) -> list[str]:
        result = []
        for members_page in self._call_paginated(method='conversations.members', channel=channel_id):
            result += members_page['members']
        return result

    def get_teams(self) -> dict:
        teams = self._call(method='admin.teams.list')
        return teams['teams']
