# coding: utf-8

import logging
import os
from collections import defaultdict

from django.conf import settings
from telethon.errors import UserAlreadyParticipantError
from telethon.tl import types, functions

from tasha.external.telethon.client import TelegramClient
from tasha.external.telethon.session import SQLiteSession
from tasha.lib.telegram import check_user_addition_in_updates

logger = logging.getLogger(__name__)
_global_clients = defaultdict(type(None))


async def _create_client():
    api_id = int(os.environ['API_ID'])
    api_hash = os.environ['API_HASH']

    tg_client = TelegramClient(
        SQLiteSession(''), api_id, api_hash,
        use_ipv6=True, receive_updates=False, sequential_updates=True,
        request_retries=10, flood_sleep_threshold=120,
    )
    client = await tg_client.start(bot_token=settings.TELEGRAM_TOKEN)

    return client


async def get_client() -> TelegramClient:
    global_client = _global_clients['client']
    if global_client is None:
        global_client = await _create_client()
        _global_clients['client'] = global_client
    return global_client


async def try_ban_member_in_chat(chat_obj, user_obj):
    client = await get_client()
    input_user = await client.get_entity(user_obj.id)

    if hasattr(chat_obj, 'access_hash') and chat_obj.access_hash:
        input_channel = types.InputChannel(chat_obj.id, chat_obj.access_hash)
        rights = types.ChatBannedRights(until_date=None, view_messages=True)
        return await client(functions.channels.EditBannedRequest(input_channel, input_user, rights))
    else:
        return await client(functions.messages.DeleteChatUserRequest(chat_obj.id, input_user))


async def try_unban_member_in_chat(chat_id, chat_title, user_id, username):
    client = await get_client()
    input_user = await client.get_input_entity(user_id, q=username)
    input_chat = await client.get_input_entity(chat_id, q=chat_title)
    rights = types.ChatBannedRights(until_date=None)
    try:
        await client(functions.channels.EditBannedRequest(input_chat, input_user, rights))
    except TypeError as err:
        logger.error(
            'Error while trying to unban %s:%s from %s:%s. %s',
            username, user_id, chat_title, chat_id, err.args,
        )


async def _get_user_by_username(client, username):
    try:
        user = await client.get_entity(username)
    except ValueError:
        user = None
    return user


async def get_user_by_username(username):
    client = await get_client()
    result = await _get_user_by_username(client, username)
    return result


async def grant_user_admin_rights(chat_obj, username):
    chat_dict = chat_obj.to_dict()
    if chat_dict['_'] != 'Channel':
        return  # В нашем случае права можно адекватно выдавать только в супергруппах

    client = await get_client()
    user_entity = await _get_user_by_username(client, username)
    input_user = types.InputUser(user_entity.id, user_entity.access_hash)
    input_channel = types.InputChannel(chat_dict['id'], chat_dict['access_hash'])

    admin_rights = chat_dict.get('admin_rights') or {}
    can_add_admins = admin_rights.get('add_admins', False) or chat_dict.get('creator', False)
    if not can_add_admins:
        logger.info('Not enough permissions to grant admin rights for chat with id=%s', chat_dict['id'])
        return

    admins = [
        user.username
        async for user in
        client.iter_participants(chat_obj, filter=types.ChannelParticipantsAdmins, aggressive=True)
    ]
    if username in admins:
        return  # Бот уже в админах

    admin_rights = types.ChatAdminRights(
        change_info=True,
        post_messages=None,  # Не могут быть выставлены для супергрупп.
        edit_messages=None,  # Аналогично. https://github.com/LonamiWebs/Telethon/issues/490
        delete_messages=True,
        ban_users=True,
        invite_users=True,
        pin_messages=True,
        add_admins=True,
    )
    await client(functions.channels.EditAdminRequest(input_channel, input_user, admin_rights))


async def try_add_user_to_chat(chat_obj, username, make_admin=False, is_bot=False, announce_message=None):
    chat_dict = chat_obj.to_dict()
    group_type = chat_dict['_']
    assert group_type in ['Chat', 'Channel']

    client = await get_client()
    user_entity = await _get_user_by_username(client, username)
    input_user = types.InputUser(user_entity.id, user_entity.access_hash)
    input_channel = None
    if group_type == 'Channel':
        input_channel = types.InputChannel(chat_dict['id'], chat_dict.get('access_hash', None))
    is_broadcast = chat_dict.get('broadcast', False)
    admin_rights = chat_dict.get('admin_rights') or {}
    can_add_admins = admin_rights.get('add_admins', False) or chat_dict.get('creator', False)

    if is_broadcast and is_bot and not (can_add_admins and make_admin):
        raise ValueError('Cannot add bot to channel without making it admin')

    updates = None
    try:
        if group_type == 'Chat':
            updates = await client(functions.messages.AddChatUserRequest(chat_dict['id'], input_user, fwd_limit=1000))
        elif group_type == 'Channel' and not is_broadcast:
            updates = await client(functions.channels.InviteToChannelRequest(input_channel, [input_user]))
    except UserAlreadyParticipantError:
        pass
    except Exception:
        logger.exception('Could not add user "%s" to chat with id=%s', username, chat_dict['id'])
        raise
    else:
        if updates is not None and check_user_addition_in_updates(updates, [user_entity.id]) and announce_message:
            if not is_broadcast:  # в канале никто про бота не узнает, а вот подписчиков отпугнёт
                await client.send_message(chat_obj, announce_message)

    if make_admin and group_type == 'Channel' and can_add_admins:
        admin_rights = types.ChatAdminRights(
            change_info=True,
            post_messages=None,  # Не могут быть выставлены для супергрупп.
            edit_messages=None,  # Аналогично. https://github.com/LonamiWebs/Telethon/issues/490
            delete_messages=True,
            ban_users=True,
            invite_users=True,
            pin_messages=(None if is_broadcast else True),
            add_admins=True,
        )
        await client(functions.channels.EditAdminRequest(input_channel, input_user, admin_rights))
