# coding: utf-8

import asyncio
import functools
import logging

# Для синхронного выполнения вызовов telethon
from telethon import sync  # noqa F401
from telethon.tl import functions, types
from telethon.errors import UserAlreadyParticipantError
from telethon.errors.rpcerrorlist import InviteHashInvalidError

from tasha.external.telethon.api import get_client
from tasha.lib.telegram import check_user_addition_in_updates
from tasha.models import TgGroupInfo, TelegramAccount

logger = logging.getLogger(__name__)


def get_client_sync():
    event_loop = asyncio.get_event_loop()
    async_client = asyncio.gather(get_client(prefix='sync_'))
    clients = event_loop.run_until_complete(async_client)
    client = clients[0]
    sync.syncify(client)
    return client


def _do_without_sqlite_caching(func):

    @functools.wraps(func)
    def decorator(*args, **kargs):
        client = get_client_sync()
        # Отключим кеширование, иначе sqlite3.OperationalError: database is locked
        old_value, client.session.save_entities = client.session.save_entities, False
        try:
            return func(client, *args, **kargs)
        finally:
            client.session.save_entities = old_value

    return decorator


@_do_without_sqlite_caching
def check_already_participants(client, invite_link):
    """
    Проверяет есть ли наши боты в чате по ссылке или нет
    :param invite_link: Ссылка на чат или канал
    :return: True - если один из ботов уже в чате, False - если наших ботов в чате нет
    """
    # Получим client из асинхронной функции get_client()
    invite_hash = invite_link.split('/')[-1]
    try:
        preview = client(functions.messages.CheckChatInviteRequest(invite_hash)).to_dict()
        if preview.get('_') == 'ChatInviteAlready':
            return True
        else:
            all_bots_ids = set(TelegramAccount.objects.filter(is_tasha=True).values_list('telegram_id', flat=True))
            all_participants_ids = set(p['id'] for p in preview['participants'])
            return bool(all_bots_ids & all_participants_ids)
    except InviteHashInvalidError:
        # Попробуем использовать invite_hash в качестве ссылки на публичный канал
        # В публичных чатах список участников видит только админ, проверить по участникам не можем
        # Проверим наличие id этого канала в нашей базе
        channel = client.get_entity(invite_hash)
        return TgGroupInfo.objects.filter(telegram_id=channel.id).exists()


@_do_without_sqlite_caching
def join_chat(client, invite_link):
    invite_hash = invite_link.split('/')[-1]
    try:
        result = client(functions.messages.ImportChatInviteRequest(invite_hash))
    except InviteHashInvalidError:
        result = client(functions.channels.JoinChannelRequest(invite_hash))
    except UserAlreadyParticipantError:
        return ('already_participant', None)

    chat_obj = result.chats[0]
    return ('success', chat_obj)


@_do_without_sqlite_caching
def create_supergroup(client, title):
    bot_self = client(functions.users.GetFullUserRequest(types.InputUserSelf()))
    bot_username = bot_self.to_dict()['user']['username']
    about = 'supergroup created by %s' % bot_username
    return client(functions.channels.CreateChannelRequest(title, about, True, True)).chats[0]


def get_user_object_by_id_and_username(client, user_id, username, phones):
    if username:
        username = username.lower()
        found = client(functions.contacts.SearchRequest(username, 1)).to_dict()
        users = found['users']
        if users:
            for user in users:
                if user['username'].lower() == username:
                    return user
    if phones:
        input_phones = []
        for phone in phones:
            first_name = 'name-%s' % phone
            last_name = 'surname-%s' % phone
            ic = types.InputPhoneContact(0, phone, first_name, last_name)
            input_phones.append(ic)
        users = client(functions.contacts.ImportContactsRequest(input_phones)).to_dict()
        users = users['users']
        for user in users:
            if user['id'] == user_id:
                return user
    raise ValueError('Can\'t get user_object')


@_do_without_sqlite_caching
def invite_user_to_group(client, chat_obj, user_id=None, username=None, phones=None, make_admin=False, is_bot=False, announce_message=None):
    chat_dict = chat_obj.to_dict()
    group_type = chat_dict['_']
    assert group_type in ['Chat', 'Channel']

    entity = get_user_object_by_id_and_username(client, user_id, username, phones)
    input_user = types.InputUser(entity['id'], entity['access_hash'])
    input_channel = None
    if group_type == 'Channel':
        input_channel = types.InputChannel(chat_dict['id'], chat_dict.get('access_hash', None))
    is_broadcast = chat_dict.get('broadcast', False)
    admin_rights = chat_dict.get('admin_rights') or {}
    can_add_admins = admin_rights.get('add_admins', False) or chat_dict.get('creator', False)

    if is_broadcast and is_bot and not (can_add_admins and make_admin):
        raise ValueError('Cannot add bot to channel without making it admin')

    updates = None
    try:
        if group_type == 'Chat':
            updates = client(functions.messages.AddChatUserRequest(chat_dict['id'], input_user, fwd_limit=1000))
        elif group_type == 'Channel' and not is_broadcast:
            updates = client(functions.channels.InviteToChannelRequest(input_channel, [input_user]))
    except UserAlreadyParticipantError:
        pass
    except Exception:
        logger.exception('Could not add user "%s" to chat with id=%s', username, chat_dict['id'])
        raise
    else:
        if updates is not None and check_user_addition_in_updates(updates, [entity['id']]) and announce_message:
            if not is_broadcast:  # в канале никто про бота не узнает, а вот подписчиков отпугнёт
                client.send_message(chat_obj, announce_message)

    if make_admin and can_add_admins:
        if group_type == 'Chat':
            pass  # Ничего с этим не поделать. У нас нет прав на добавление админов в чаты.
            # А если вдруг появились, то группа будет преобразована в супергруппу
            # А если мы создатели этот чат, то это супергруппа с самого начала
        elif group_type == 'Channel':
            admin_rights = types.ChatAdminRights(
                change_info=True,
                post_messages=None,  # Не могут быть выставлены для супергрупп.
                edit_messages=None,  # Аналогично. https://github.com/LonamiWebs/Telethon/issues/490
                delete_messages=True,
                ban_users=True,
                invite_users=True,
                pin_messages=(None if is_broadcast else True),
                add_admins=True,
            )
            client(functions.channels.EditAdminRequest(input_channel, input_user, admin_rights))


@_do_without_sqlite_caching
def send_message(client, chat_obj, message):
    bot_username = client.get_me().username.lower()
    logger.info(f'Sending message {message} from bot {bot_username} to chat {chat_obj.title}')
    client.send_message(chat_obj, message)
