# coding: utf-8

import logging

from telethon import utils
from telethon.crypto import AuthKey
from telethon.sessions import abstract
from telethon.sessions.sqlite import SQLiteSession as BaseSQLiteSession
from telethon.sessions.memory import _SentFileType
from telethon.tl import TLObject
from telethon.tl.types import (
    InputPeerUser, InputPeerChat, InputPeerChannel,
    InputDocument, InputPhoto,
    PeerUser, PeerChat, PeerChannel,
)

from tasha import models


logger = logging.getLogger(__name__)


def _entity_values_to_row(id, hash, username, phone, name):
    return {
        'telegram_id': id,
        'access_hash': hash,
        'username': username,
        'phone': phone,
        'name': name,
    }


def _entity_to_row(e):
    if not isinstance(e, TLObject):
        return
    try:
        p = utils.get_input_peer(e, allow_self=False)
        marked_id = utils.get_peer_id(p)
    except TypeError:
        return

    if isinstance(p, (InputPeerUser, InputPeerChannel)):
        if not p.access_hash:
            return
        else:
            p_hash = p.access_hash
    elif isinstance(p, InputPeerChat):
        p_hash = 0
    else:
        return

    username = getattr(e, 'username', None) or None
    if username is not None:
        username = username.lower()
    phone = getattr(e, 'phone', None)
    name = utils.get_display_name(e) or None
    return _entity_values_to_row(
        marked_id, p_hash, username, phone, name
    )


def _entities_to_rows(tlo):
    if not isinstance(tlo, TLObject) and utils.is_list_like(tlo):
        entities = tlo
    else:
        entities = []
        if hasattr(tlo, 'chats') and utils.is_list_like(tlo.chats):
            entities.extend(tlo.chats)
        if hasattr(tlo, 'users') and utils.is_list_like(tlo.users):
            entities.extend(tlo.users)

    rows = []  # Rows to add (id, hash, username, phone, name)
    for e in entities:
        row = _entity_to_row(e)
        if row:
            rows.append(row)
    return rows


class SQLiteSession(BaseSQLiteSession):
    @property
    def server_address(self):
        server_address = super().server_address
        if server_address:
            server_address = server_address.replace('[', '').replace(']', '')

        return server_address


class TelethonPostgresSession(abstract.Session):
    def __init__(self, session_name):
        super(TelethonPostgresSession, self).__init__()
        self.session_name = session_name

    def set_dc(self, dc_id, server_address, port):
        models.TelethonSession.objects.update_or_create(
            session_name=self.session_name,
            defaults={
                'dc_id': dc_id,
                'server_address': server_address,
                'port': port,
            },
        )

    @property
    def server_address(self):
        try:
            return models.TelethonSession.objects.get(
                session_name=self.session_name
            ).server_address
        except models.TelethonSession.DoesNotExist:
            return None

    @property
    def port(self):
        try:
            return models.TelethonSession.objects.get(
                session_name=self.session_name
            ).port
        except models.TelethonSession.DoesNotExist:
            return None

    @property
    def auth_key(self):
        try:
            key_content = models.TelethonSession.objects.get(
                session_name=self.session_name
            ).auth_key
            if key_content is None:
                return None
            return AuthKey(data=bytes(key_content))
        except models.TelethonSession.DoesNotExist:
            return None

    @auth_key.setter
    def auth_key(self, value):
        if value is None:
            return
        models.TelethonSession.objects.filter(
            session_name=self.session_name
        ).update(auth_key=bytes(value.key))

    def save(self):
        pass

    def delete(self):
        pass

    def close(self):
        pass

    def process_entities(self, tlo):
        rows = _entities_to_rows(tlo)
        for row in rows:
            telegram_id = row.pop('telegram_id')
            (_, created) = models.TelethonEntity.objects.update_or_create(
                session_name=self.session_name,
                telegram_id=telegram_id,
                defaults=row
            )
            # if created:
            #     logger.debug('entity created %s, %r', telegram_id, row)
            # else:
            #     logger.debug('entity updated %s, %r', telegram_id, row)

    def get_entity_rows_by_phone(self, phone):
        try:
            obj = models.TelethonEntity.objects.get(
                session_name=self.session_name, phone=phone
            )
            return (obj.telegram_id, obj.access_hash)
        except models.TelethonEntity.DoesNotExist:
            return None

    def get_entity_rows_by_username(self, username):
        try:
            obj = models.TelethonEntity.objects.get(
                session_name=self.session_name, username=username
            )
            return (obj.telegram_id, obj.access_hash)
        except models.TelethonEntity.DoesNotExist:
            return None

    def get_entity_rows_by_name(self, name):
        try:
            obj = models.TelethonEntity.objects.get(
                session_name=self.session_name, name=name
            )
            return (obj.telegram_id, obj.access_hash)
        except models.TelethonEntity.DoesNotExist:
            return None

    def get_entity_rows_by_id(self, id_, exact=True):
        if exact:
            ids = (id_,)
        else:
            ids = (
                utils.get_peer_id(PeerUser(id_)),
                utils.get_peer_id(PeerChat(id_)),
                utils.get_peer_id(PeerChannel(id_))
            )
        obj = models.TelethonEntity.objects.filter(
            session_name=self.session_name, telegram_id__in=ids
        ).first()
        if obj is None:
            return obj
        return (obj.telegram_id, obj.access_hash)

    def get_input_entity(self, key):
        try:
            if key.SUBCLASS_OF_ID in (0xc91c90b6, 0xe669bf46, 0x40f202fd):
                # hex(crc32(b'InputPeer', b'InputUser' and b'InputChannel'))
                # We already have an Input version, so nothing else required
                return key
            # Try to early return if this key can be casted as input peer
            return utils.get_input_peer(key)
        except (AttributeError, TypeError):
            # Not a TLObject or can't be cast into InputPeer
            if isinstance(key, TLObject):
                key = utils.get_peer_id(key)
                exact = True
            else:
                exact = not isinstance(key, int) or key < 0

        result = None
        if isinstance(key, str):
            phone = utils.parse_phone(key)
            if phone:
                result = self.get_entity_rows_by_phone(phone)
            else:
                username, _ = utils.parse_username(key)
                if username:
                    result = self.get_entity_rows_by_username(username)

        elif isinstance(key, int):
            result = self.get_entity_rows_by_id(key, exact)

        if not result and isinstance(key, str):
            result = self.get_entity_rows_by_name(key)

        if result:
            i, h = result  # unpack resulting tuple
            i, k = utils.resolve_id(i)  # removes the mark and returns kind
            if k == PeerUser:
                return InputPeerUser(i, h)
            elif k == PeerChat:
                return InputPeerChat(i)
            elif k == PeerChannel:
                return InputPeerChannel(i, h)
        else:
            raise ValueError('Could not find input entity with key ', key)

    def cache_file(self, md5_digest, file_size, instance):
        if not isinstance(instance, (InputDocument, InputPhoto)):
            raise TypeError('Cannot cache %s instance' % type(instance))
        obj = models.TelethonSentFile(
            session_name=self.session_name,
            md5_digest=md5_digest,
            file_size=file_size,
            file_type=_SentFileType.from_type(type(instance)).value,
            file_id=instance.id,
            access_hash=instance.access_hash,
        )
        obj.save()

    def get_file(self, md5_digest, file_size, cls):
        obj = models.TelethonSentFile.objects.filter(
            session_name=self.session_name,
            md5_digest=md5_digest,
            file_size=file_size,
            file_type=_SentFileType.from_type(cls).value,
        )
        if obj is None:
            return None
        return (obj.file_id, obj.access_hash)
