# coding: utf-8
"""
Usage:

    @cache_storage.memoize_decorator(300)
    def foo(x):
        return 42

The value 42 will be cached in redis or memory.


    def bar(y, z):
        return 5

    cache_storage.memoize(300)(bar, y, z)

The value 5 will be cached in redis or memory.

"""

from __future__ import unicode_literals

import collections
import datetime
import functools
import hashlib
import json
import logging
import time

from django.conf import settings
from django.core.cache import cache


logger = logging.getLogger(__name__)
DEFAULT_TIMEOUT = 60  # seconds
EXPIRED_VALUE = 'EXPIRED_VALUE'
_cached_objects = {}


class _CacheEncoder(json.JSONEncoder):
    def _transform_object(self, o):
        if isinstance(o, (str, bytes, bytearray)):
            return o
        elif isinstance(o, collections.Mapping):
            return collections.OrderedDict(sorted(
                [(k, self._transform_object(v)) for k, v in o.items()],
                key=lambda k: str(self._transform_object(k[0]))
            ))
        elif isinstance(o, collections.Sequence):
            return sorted(map(str, map(self._transform_object, o)))
        elif isinstance(o, datetime.datetime):
            return str(o)

        return o

    def encode(self, o):
        return super().encode(self._transform_object(o))


def memoize_decorator(fresh_time=DEFAULT_TIMEOUT, cache_none=True, cache_prefix=''):
    def _decorator(func_or_method):
        @functools.wraps(func_or_method)
        def _wrapper(*args, **kwargs):
            key = cache_prefix + _key(func_or_method, _CacheEncoder, *args, **kwargs)
            current_time = time.time()
            try:
                (cached_time, value) = _get(key)
                if current_time - cached_time > fresh_time:
                    logger.debug('Invalid cached value for %s (%r, %r)', func_or_method, args, kwargs)
                    raise ValueError()
            except (ValueError, KeyError):
                logger.debug('Calculate value for %s (%r, %r)', func_or_method, args, kwargs)
                value = func_or_method(*args, **kwargs)
                if cache_none or value is not None:
                    set_key_value(key, value, fresh_time)
                else:
                    logger.debug('Calculated value is None and cache_none=False')
            return value
        return _wrapper
    return _decorator


def memoize_decorator_async(fresh_time=DEFAULT_TIMEOUT, cache_none=True, cache_prefix=''):
    def _decorator(func_or_method):
        @functools.wraps(func_or_method)
        async def _wrapper(*args, **kwargs):
            key = cache_prefix + _key(func_or_method, _CacheEncoder, *args, **kwargs)
            current_time = time.time()
            try:
                (cached_time, value) = _get(key)
                if current_time - cached_time > fresh_time:
                    logger.debug('Invalid cached value for %s (%r, %r)', func_or_method, args, kwargs)
                    raise ValueError()
            except (ValueError, KeyError):
                logger.debug('Calculate value for %s (%r, %r)', func_or_method, args, kwargs)
                value = await func_or_method(*args, **kwargs)
                if cache_none or value is not None:
                    set_key_value(key, value, fresh_time)
                else:
                    logger.debug('Calculated value is None and cache_none=False')
            return value
        return _wrapper
    return _decorator


def memoize(fresh_time=DEFAULT_TIMEOUT, cache_none=True, cache_prefix=''):
    def _internal(func_or_method, *args, **kwargs):
        return memoize_decorator(fresh_time, cache_none, cache_prefix)(func_or_method)(*args, **kwargs)
    return _internal


def _build_key_from_args(cls, *args, **kwargs):
    if not args and not kwargs:
        return ''
    hash_content = json.dumps([args, kwargs], sort_keys=True, cls=cls)
    return hashlib.sha1(hash_content.encode('utf-8')).hexdigest()


def _func_path(func_or_method):
    """Return absolute `func_or_method` path.

    For function `foo()` in module `proj.mod` it will be `proj.mod.foo`.
    For method `m()` of class `A` of the same module it will be
    `proj.mod.A.m`.

    It does not work for nested classes and functions.

    :param func_or_method: Function or method.
    :return: String containing full `func_or_method` path.
    """
    params = []
    if hasattr(func_or_method, '__module__'):
        params.append(func_or_method.__module__)
    if hasattr(func_or_method, 'im_class'):
        params.append(func_or_method.im_class.__name__)
    if hasattr(func_or_method, '__name__'):
        params.append(func_or_method.__name__)
    return '.'.join(filter(None, params))


def _key(func_or_method, cls, *args, **kwargs):
    func_key = _func_path(func_or_method)
    args_key = _build_key_from_args(cls, *args, **kwargs)

    return func_key + args_key


def _get(key):
    logger.debug('Cache _GET key [%r]', key)
    if settings.IS_REDIS_CACHE:
        value = cache.get(key, EXPIRED_VALUE)
        if value == EXPIRED_VALUE:
            raise KeyError()
        else:
            return value
    else:
        return _cached_objects[key]


def set_key_value(key, value, timeout=None):
    logger.debug('Cache SET key [%r]', key)
    if settings.IS_REDIS_CACHE:
        cache.set(key, (time.time(), value), timeout)
    else:
        _cached_objects[key] = (time.time(), value)


def get_key_value(key):
    logger.debug('Cache GET key [%r]', key)
    try:
        (cached_time, value) = _get(key)
        return value
    except KeyError:
        return EXPIRED_VALUE
