from __future__ import unicode_literals
from ids.registry import registry
import ids

from tasha.core import dbproxy
from tasha.config import settings
from tasha.lib.coro_wrapper import coro_run
from tasha.external import staff

import click

CHANNEL_ID = -1001755799115  # TG id чата "Информирование Руководителей"

persons = registry.get_repository(
    'staff', 'person',
    user_agent='uhura', oauth_token=settings.ROBOT_TOKEN, retries=5, timeout=settings.STAFF_REQUEST_TIMEOUT
)


async def download_chiefs(cnt):
    """
        The function is getting a list of user's chiefs from staff,
        according to a given number of chiefs these users have
    """
    params = {
        '_fields': ','.join((
            'id',
            'chiefs.login',
            'login'
        )),
        'official.is_dismissed': 'false',
        'official.is_robot': 'false',
        'official.affiliation': 'yandex',
        '_sort': 'id',
        '_limit': settings.STAFF_PAGE_LIMIT,
    }
    logins = []
    last_id = 0
    query_template = f'chiefs==size({cnt}) and id > %d'
    while True:
        params['_query'] = query_template % last_id
        print(params['_query'])
        raw_persons = list(staff.persons.getiter(lookup=params))
        if len(raw_persons) == 0:
            break
        for person in raw_persons:
            for c in person['chiefs']:
                logins.append(c['login'])
        last_id = raw_persons[-1]['id']
    uniques = set(logins)
    return list(uniques)


async def download_persons(cnt):
    """
        The function is getting user's data from staff,
        according to a given number of chiefs these users have
    """
    params = {
        '_fields': ','.join((
            'id',
            'chiefs.login',
            'login'
        )),
        'official.is_dismissed': 'false',
        'official.is_robot': 'false',
        'official.affiliation': 'yandex',
        '_sort': 'id',
        '_limit': settings.STAFF_PAGE_LIMIT,
    }
    logins = []
    last_id = 0
    query_template = f'chiefs==size({cnt}) and id > %d'
    while True:
        params['_query'] = query_template % last_id
        raw_persons = list(staff.persons.getiter(lookup=params))
        if len(raw_persons) == 0:
            break
        for person in raw_persons:
            logins.append(person['login'])
        last_id = raw_persons[-1]['id']
    uniques = set(logins)
    return list(uniques)


async def get_tops():
    """
        Comparing the list of people with 0, 1 or 2 chiefs as left arr with
        the list of chiefs of people with up to 4 chiefs as right to make sure
        the result will contain only chiefs
    """
    res = []
    left = await download_persons(0) + await download_persons(1) + await download_persons(2)
    right = await download_chiefs(4) + await download_chiefs(3) + await download_chiefs(2) + await download_chiefs(1)
    for person in left:
        if person in right:
            res.append(person)
    return res


@click.command('compare_users')
@coro_run
async def compare_users():
    """
        The command is used to compare the list of users from chat with
        a list of high tier chiefs taken from staff. The chief list parameters are set
        in functions above.
    """
    to_invite = []
    to_kick = []
    db_proxy = dbproxy.DBProxy()
    try:
        await db_proxy.connect()
        row = await db_proxy.get_top_channels({'CHANNEL_ID': CHANNEL_ID})
        ids_from_chat = row[0]['members']
        staffs_from_chat = []
        staffs_rows = await db_proxy.get_staff_usernames(ids_from_chat)
        for rw in staffs_rows:
            staffs_from_chat.append(rw['staff_username'])

        tops = await get_tops()

        # looking for people who are appropriate but not in chat yet
        for login in tops:
            if login in staffs_from_chat:
                continue
            else:
                to_invite.append(login)

        # looking for people who are not appropriate anymore
        for login in staffs_from_chat:
            if login in tops:
                continue
            else:
                to_kick.append(login)

        _to_kick = list(set(to_kick))  # getting unique values
        _to_invite = list(set(to_invite))  # getting unique values

        print('people to kick: ', _to_kick, 'people to invite: ', _to_invite, sep='\n', )
        return _to_kick, _to_invite
    finally:
        await db_proxy.close()
