import logging
import time
from typing import Iterator

import yenv
import yt.wrapper
from django.conf import settings

from tasha.core import TashaUnit
from tasha.core.dbproxy import DBProxy
from tasha.external.slack import TashaSlackAPI, TashaSlackAPIException
from tasha.lib.export.base import BaseExport


@yt.wrapper.yt_dataclass
class SlackGroupRow:
    team_id: str
    team_name: str
    channel_id: str
    channel_title: str
    channel_description: str
    channel_type: str
    channel_created: int
    channel_members: list[str]
    channel_admins: list[str]


class SlackExport(BaseExport):
    yt_client: yt.wrapper.YtClient
    yt_table_path: str
    yt_table_created: bool = False

    def __init__(self, tu: TashaUnit):
        super().__init__(tu)
        self.log = logging.getLogger(__name__)
        self.bot_api = TashaSlackAPI(token=settings.SLACK_EXPORT_BOT_TOKEN, page_size=settings.SLACK_EXPORT_PAGE_SIZE)
        self.user_api = TashaSlackAPI(token=settings.SLACK_EXPORT_USER_TOKEN, page_size=settings.SLACK_EXPORT_PAGE_SIZE)

    async def export(self, db_proxy: DBProxy):
        self.log.info('Handle slack export')
        self._init_yt()
        for channels_chunk in self._get_slack_channels_chunks():
            self._export_slack_channels_to_yt(channels_chunk)
        self._rotate_obsolete_yt_tables()
        self.log.info('Handle slack export done')

    def _init_yt(self):
        self.yt_table_path = '{directory}/{table}_{timestamp}'.format(
            directory=settings.SLACK_EXPORT_YT_TABLE_DIR,
            table=settings.SLACK_EXPORT_YT_TABLE_NAME,
            timestamp=int(time.time()),
        )
        self.yt_client = yt.wrapper.YtClient(settings.SLACK_EXPORT_YT_PROXY, token=settings.YT_TOKEN)

    def _get_slack_channels_chunks(self) -> Iterator[list[SlackGroupRow]]:
        data = []

        if yenv.type != 'production':
            teams = settings.SLACK_EXPORT_TEAMS
        else:
            teams = {team['id']: team['name'] for team in self.user_api.get_teams()}

        for team_id, team_name in teams.items():
            try:
                users = self.bot_api.get_users_id_login_map(team_id=team_id)
            except TashaSlackAPIException as e:
                self.log.warning('Got slack error while retrieving users for team `%s`: %s', team_id, e)
                continue

            try:
                channels = self.bot_api.get_channels(
                    team_id=team_id,
                    types=settings.SLACK_EXPORT_CHANNEL_TYPES,
                    exclude_archived=True,
                )
            except TashaSlackAPIException as e:
                self.log.warning('Got slack error while retrieving channels for team `%s`: %s', team_id, e)
                continue

            for channel in channels:
                try:
                    member_ids = self.bot_api.get_channel_members(channel['id'])
                except TashaSlackAPIException as e:
                    self.log.warning(
                        'Got slack error while retrieving channel members for channel `%s`: %s', channel['id'], e
                    )
                    continue

                admins = [users[channel['creator']]] if channel['creator'] in users else []
                members = [users[member_id] for member_id in member_ids if member_id in users]

                if not members:
                    continue

                data.append(
                    SlackGroupRow(
                        channel_id=channel['id'],
                        team_id=team_id,
                        team_name=team_name,
                        channel_title=channel['name'],
                        channel_description=channel['topic']['value'],
                        channel_type='private' if channel['is_private'] else 'public',
                        channel_created=channel['created'],
                        channel_admins=admins,
                        channel_members=members,
                    ),
                )
                if len(data) == settings.SLACK_EXPORT_CHANNEL_CHUNK_SIZE:
                    yield data
                    data = []
        if data:
            yield data

    def _export_slack_channels_to_yt(self, data: list[SlackGroupRow]):
        if not self.yt_table_created:
            force_create = self.yt_table_created = True
        else:
            force_create = False

        self.log.info('Writing %d rows to YT %s start force_create=%s', len(data), self.yt_table_path, force_create)
        self.yt_client.write_table_structured(
            table=self.yt_client.TablePath(self.yt_table_path, append=not force_create),
            row_type=SlackGroupRow,
            input_stream=data,
            force_create=force_create,
        )
        self.log.info('Writing %d rows to YT %s end', len(data), self.yt_table_path)

    def _rotate_obsolete_yt_tables(self):
        existing_tables = list(
            self.yt_client.search(
                root=settings.SLACK_EXPORT_YT_TABLE_DIR,
                node_type=['table'],
                attributes=['creation_time'],
            )
        )
        if len(existing_tables) > settings.SLACK_EXPORT_YT_TABLE_COUNT:
            existing_tables_by_creation = sorted(
                existing_tables,
                key=lambda t: t.attributes['creation_time'],
            )
            self.log.info('Cleaning old tables')
            for existing_table in existing_tables_by_creation[:-settings.SLACK_EXPORT_YT_TABLE_COUNT]:
                self.log.info('Removing `%s`', existing_table)
                self.yt_client.remove(existing_table)
