import time
from abc import ABCMeta, abstractmethod

from ylog.context import log_context, get_log_context


class BaseAsyncContext(log_context, metaclass=ABCMeta):
    TAG = None
    collector = None

    def __init__(self, **kwargs):
        super().__init__(**self.collector.collect(tag=self.TAG, **kwargs))

    def __enter__(self):
        super().__enter__()
        self._after_enter()
        return self

    def __exit__(self, exc_type, exc_value, exc_traceback):
        self.exception_data = (exc_type, exc_value, exc_traceback) if exc_type else None
        self._before_exit()
        super().__exit__(exc_type, exc_value, exc_traceback)

    def _after_enter(self):
        self.start_time = time.time()

    def _before_exit(self):
        execution_time = time.time() - self.start_time
        with log_context(execution_time=1000 * execution_time):
            self.log_after_execution(get_log_context())

    @abstractmethod
    def log_after_execution(self, context):
        raise NotImplementedError()
