import collections
import logging

from django.conf import settings
from django.core.mail import EmailMultiAlternatives
from jinja2 import Environment, FileSystemLoader

import yenv

from tasha.lib import cache_storage
from tasha.lib.telegram import bot


User = collections.namedtuple('User', ['username', 'first_name', 'last_name'])
User.__new__.__defaults__ = ('',) * len(User._fields)
env = Environment(loader=FileSystemLoader(settings.MAIL_TEMPLATES_PATH))
logger = logging.getLogger(__name__)

USER_MESSAGE_SUBJECT = '[@{bot}] Unknown users have been detected in your chat {chat}'
USER_TOO_MANY_MESSAGE_SUBJECT = '[@{bot}] Too many unknown users have been detected in your chat {chat}'
DIALOG_CACHE_TIMEOUT = 24 * 3600


# TODO: remove caching here (after adding it to slackbot)
@cache_storage.memoize_decorator(fresh_time=86300)  # a bit less than 24 hours
def send_mail(to, subject, template_name, context=None):
    logger.info(
        'Mail alert. Template: %s, To: %s, Context: %s',
        template_name, to, context
    )
    if settings.MAIL_ALERT_OFF:
        return 1
    if yenv.type != 'production':
        subject = '[%s] %s' % (yenv.type.upper(), subject)

    if not isinstance(to, list):
        to = [to]
    reply_to = settings.REPLY_TO

    html_template = env.get_template(template_name + '.html')
    text_template = env.get_template(template_name + '.txt')
    html_content = html_template.render(context=context)
    text_content = text_template.render(context=context)

    message = EmailMultiAlternatives(
        subject=subject,
        body=text_content,
        from_email=settings.MAIL_FROM,
        to=to,
        reply_to=reply_to,
        bcc=[settings.BCC_EMAIL],
    )
    message.attach_alternative(html_content, 'text/html')
    message.send()
    # for caching and not alert for a long time
    return 1


def format_user_name(user):
    if isinstance(user, dict):
        user = User(**{k: v for k, v in user.items() if k in User._fields})

    print_name = f'{user.first_name or ""} {user.last_name or ""}'.strip()
    username = getattr(user, 'username', '')
    if username:
        print_name = f'{print_name} (@{username})'.strip()

    if getattr(user, 'deleted', False) and not print_name:
        print_name = 'DELETED ACCOUNT'

    return print_name


async def notify_about_not_kicked_users(bot_username, admins_emails, not_kicked_users, chat_title, chat_id, db_proxy,
                                        too_many=False, client=None):
    if not not_kicked_users:
        logger.error('No users')
        return

    if not admins_emails and client is None and not settings.IS_GOD_BOT:
        logger.error('group without admin emails %r', chat_title)
        return

    print_users = [format_user_name(user) for user in not_kicked_users]
    context = {
        'bot_username': bot_username,
        'print_users': print_users,
        'group_title': chat_title,
    }

    prev_not_kicked_users = cache_storage.get_key_value(chat_id)
    if set(print_users).issubset(set(prev_not_kicked_users)):
        logger.info('Already notified about chat %s', chat_title)
        return

    if admins_emails:
        subject_template = USER_TOO_MANY_MESSAGE_SUBJECT if too_many else USER_MESSAGE_SUBJECT
        subject = subject_template.format(bot=bot_username, chat=chat_title)

        logger.info('Trying to notify about chat %s', chat_title)
        send_mail(admins_emails, subject, 'telegram_notify', context)
    else:
        message = (
            'Внимание! В чате имеются пользователи, чьи telegram-аккаунты не являются аккаунтами '
            'действующих сотрудников Яндекса\n%s' % '\n'.join(print_users)
        )
        logger.info('Trying to send message about %s to chat %s', print_users, chat_title)
        if settings.IS_GOD_BOT:
            await bot.send_message(chat_id=chat_id, text=message)
        else:
            await client.send_message(chat_id, message)
        logger.info('Successfully sent message about %s to chat %s', print_users, chat_title)

    cache_storage.set_key_value(chat_id, print_users, DIALOG_CACHE_TIMEOUT)
    await db_proxy.register_email(not_kicked_users, chat_id)
