from dataclasses import dataclass
from datetime import date, datetime
from typing import Optional

from tasha.config import settings
from tasha.external import staff
from tasha.lib.utils import safe_getitem


def person_in_ex_yandex_money(person: dict) -> bool:
    return (
            settings.EX_YANDEX_MONEY_URL in (g['url'] for g in person['department_group']['ancestors']) or
            person['department_group']['url'] == settings.EX_YANDEX_MONEY_URL
    )


@dataclass
class StaffPerson:
    username: str
    email: Optional[str]
    quit_at: Optional[date]
    join_at: Optional[date]
    leave_at: Optional[datetime]
    organization_id: int
    affiliation: str
    is_active: bool
    accounts: set[str]

    @classmethod
    def from_staff_raw(cls, raw: dict):
        quit_at = safe_getitem(raw, ['official', 'quit_at'])
        join_at = safe_getitem(raw, ['official', 'join_at'])
        return cls(
            username=raw['login'],
            email=raw['work_email'],
            quit_at=date.fromisoformat(quit_at) if quit_at else None,
            join_at=date.fromisoformat(join_at) if join_at else None,
            leave_at=None,
            organization_id=safe_getitem(raw, ['official', 'organization', 'id']),
            affiliation=safe_getitem(raw, ['official', 'affiliation']),
            is_active=not safe_getitem(raw, ['official', 'is_dismissed']) and not person_in_ex_yandex_money(raw),
            accounts=staff.find_telegram_accounts(raw['accounts'])
        )
