import logging
from dataclasses import asdict
from datetime import datetime, timedelta
from typing import Iterable, Dict

from tasha.config import settings
from tasha.core import TashaUnit
from tasha.core.models import User
from tasha.core.services import UserService
from tasha.db.gateways.user import UserWithAccount
from tasha.external import staff
from tasha.lib.staff.staff_person import StaffPerson
from tasha.lib.staff.utils import has_changes

logger = logging.getLogger(__name__)


class StaffAccountsPull:
    """
    Класс, в котором описана логика синхронизации данных из Staff в Tasha
    - получаем список изменений со стаффа
    - берем выгрузки по username из базы
    - берем дифф
    - если сотрудник уволился то ставим таску на кик из чатов
    - если был дифф обновляем сотруика в базе
    - смотрим на изменения в аккаунтах телеги, если какие то поменялись то привязываем/отвязываем
    - ВАЖНО: не отвязываем аккаунты созданные ухурой
    """
    def __init__(self, tu: TashaUnit, user_srv: UserService):
        self.tu = tu
        self.user_srv = user_srv

    @classmethod
    async def init(cls, tu: TashaUnit):
        return cls(tu=tu, user_srv=UserService(tu))

    async def pull_staff_users(self) -> None:
        """
        Подтягивание пользователей из Staff
        - при синке важно не отвязывать аккаунты созданные ухурой created_with_uhura = True
        """
        lookup = {
            '_fields': ','.join((
                'id',
                'uid',
                'login',
                'accounts',
                'department_group.ancestors.url',
                'department_group.url',
                'official.is_dismissed',
                'official.quit_at',
                'official.join_at',
                'official.organization',
                'official.affiliation',
                'work_email'
            )),
            '_sort': 'id',
            '_limit': settings.STAFF_PAGE_LIMIT,
        }
        start_time = datetime.now() - timedelta(minutes=3)
        query_template = 'id > %d'
        last_time = await self._get_last_time()

        if last_time:
            query_template += ' and _meta.modified_at > "%s"' % last_time.strftime("%Y-%m-%dT%H:%M:%S%z")

        last_id = 0
        while True:
            logger.info('request users with id: %s, time: %s', last_id, last_time.strftime("%Y-%m-%dT%H:%M:%S%z"))
            lookup['_query'] = query_template % last_id
            raw_persons = list(staff.persons.getiter(lookup=lookup).first_page)
            if len(raw_persons) == 0:
                break

            last_id = raw_persons[-1]['id']
            # конвертируем сырой ответ стаффа в модельку
            persons: [StaffPerson] = [StaffPerson.from_staff_raw(person) for person in raw_persons]
            # получаем список логинов
            usernames = [person.username for person in persons]
            # запрашиваем из быза пользователей с этими логинами
            username_to_user = await self._get_db_users_cache(usernames=usernames)
            # запрашиваем привязку аккаунтов к этим логинам
            username_to_accounts = await self._get_db_users_accounts(usernames=usernames)

            # Заполняем всех пользователей для синхронизации аккаунтов
            users_and_persons = []
            for person in persons:
                username = person.username
                if username in username_to_user:
                    user = username_to_user[username]
                    # пользователь уволился, кикаем его ото всюду
                    if user.is_active and not person.is_active:
                        person.leave_at = datetime.now()
                        self.tu.add_job('kick_leaved_user', username=username)
                    # пользователь терминатор, достаем его из бан листов
                    if not user.is_active and person.is_active:
                        self.tu.add_job('unban_user', username=username)
                    if has_changes(user, person):
                        await self.tu.user.update(user.id, **asdict(person))
                else:
                    user = await self.tu.user.create(**asdict(person))
                users_and_persons.append((user, person))
            # Проходимся по всем старым/новым пользователям и синкаем телеграмы
            for user, person in users_and_persons:
                user_accounts = username_to_accounts[user.username] if user.username in username_to_accounts else None
                await self.user_srv.update_usernames(user, user_accounts,  person.accounts)

        await self.tu.sync_time.update(name='import_staff', last_success_start=start_time)

    async def _get_db_users_cache(self, usernames: Iterable[str]) -> Dict[str, User]:
        db_users = await self.tu.user.get_users_by_usernames(usernames)
        return {user.username: user for user in db_users}

    async def _get_db_users_accounts(self, usernames: Iterable[str]) -> Dict[str, list[UserWithAccount]]:
        db_users_accounts = await self.tu.user.get_users_accounts(usernames)
        res = {}
        for user in db_users_accounts:
            if user.username not in res:
                res[user.username] = []
            res[user.username].append(user)
        return res

    async def _get_last_time(self):
        last_time = None
        sync_time = await self.tu.sync_time.get(name='import_staff')
        if not sync_time:
            await self.tu.sync_time.create(name='import_staff')
        else:
            last_time = sync_time.last_success_start

        return last_time
