import logging

import asyncio
from aiotg import Bot as AioTgBot
from aiotg.bot import BotApiError  # noqa
from aiotg.chat import Chat  # noqa

from django.conf import settings

from tasha import models
from tasha.external import staff


ADD_BOT_ANNOUNCE_MESSAGE = """Привет! Сейчас в чат был добавлен бот {bot_username}.
Не стоит пугаться, это позволит мне удалять из чата посторонних людей надёжнее и оперативнее.
Вопросы можно задавать в Этушке: https://clubs.at.yandex-team.ru/tools/15750
"""


async def get_chat(chat_id: int) -> dict:
    """
    Достаёт детали чата и пермишны бота в чате
    """
    chat_info = await bot.get_chat(chat_id=chat_id)
    return chat_info['result']


async def is_channel(chat_id: int) -> bool:
    chat_info = await get_chat(chat_id)
    return chat_info['type'] == 'channel'


class Bot(AioTgBot):
    """
    Прослойка, через которую можно вызвать любой метод в bot API Телеграма
    """
    def __init__(self, token=None, custom_message_handler=None):
        self.custom_message_handler = custom_message_handler
        super().__init__(token or settings.TELEGRAM_TOKEN)

    def __getattr__(self, method_name):
        method_name = ''.join(method_name.split('_'))

        def get(self, **kwargs):
            return self.api_call(method_name, **kwargs)

        return get.__get__(self)

    async def send_message(self, chat_id, text, **options):
        if await is_channel(chat_id):
            logging.warning('Avoid writing to channels. Message: %s', text)
            return

        return super().send_message(chat_id, text, **options)

    def _process_message(self, message):
        if self.custom_message_handler:
            asyncio.ensure_future(
                self.custom_message_handler(message)
            )
        return super()._process_message(message)


def get_person_data_by_telegram_id(telegram_id):
    if not telegram_id:
        return None
    try:
        user_obj = models.User.objects.get(telegram_accounts__telegram_id=telegram_id)
    except models.User.DoesNotExist:
        return None
    person_data = staff.get_person_data_by_username(user_obj.username)
    if person_data is not None:
        return person_data

    user_phone = user_obj.phone
    return staff.get_person_data_by_userphone(user_phone)


def get_user_phones_by_telegram_id(telegram_id):
    person_data = get_person_data_by_telegram_id(telegram_id)
    phones = []
    if person_data:
        for phone in person_data['phones']:
            phones.append(phone['number'])
    return phones


def validate_telegram_id(telegram_id):
    tg_username = models.TelegramAccount.objects.filter(telegram_id=telegram_id).first()
    return tg_username and tg_username.user_id and tg_username.user.is_active


def check_user_addition_in_updates(updates, user_ids):
    for event in updates.to_dict()['updates']:
        if event['_'] not in ('UpdateNewMessage', 'UpdateNewChannelMessage'):
            continue
        if not event.get('message'):
            continue
        if not event['message'].get('action'):
            continue
        if event['message']['action']['_'] != 'MessageActionChatAddUser':
            continue
        users = event['message']['action']['users']
        if set(user_ids) <= set(users):
            return True
    return False


bot = Bot()
