import logging

from django.conf import settings
from django.core.exceptions import PermissionDenied
from django.core.handlers.wsgi import WSGIRequest

from tasha.utils import get_tvm_client

logger = logging.getLogger(__name__)


class TVMMiddleware:
    def check_service_ticket(self, ticket):
        client = get_tvm_client()
        parsed_ticket = client.parse_service_ticket(ticket)
        return parsed_ticket is not None and parsed_ticket.src in settings.ALLOWED_TVM_APPS

    def __init__(self, get_response):
        self.get_response = get_response

    def __call__(self, request: WSGIRequest):
        should_check_service_ticket = True
        for prefix in settings.UNPROTECTED_PATHS:
            if request.path.startswith(prefix):
                should_check_service_ticket = False
        if should_check_service_ticket:
            tvm_ticket = request.META.get('HTTP_X_YA_SERVICE_TICKET')
            if not tvm_ticket or not self.check_service_ticket(tvm_ticket):
                logger.info('Bad tvm ticker received, ticket = %s', tvm_ticket)
                raise PermissionDenied()
        response = self.get_response(request)
        return response
