from typing import Iterable
from django.db import models, transaction
from django.contrib.postgres import fields

from tasha.constants import ACTION, CHAT_TYPE


class OAuthToken(models.Model):
    class Meta:
        app_label = 'tasha'
        db_table = 'tasha_oauthtoken'

    token = models.CharField(max_length=128)

    def __str__(self):
        return f'{self.token}'


class SlackStatistics(models.Model):
    class Meta:
        app_label = 'tasha'
        db_table = 'tasha_slackstatistics'

    statistics = fields.JSONField(null=False)
    updated_at = models.DateTimeField(auto_now=True, db_index=True)


class SyncTime(models.Model):
    name = models.TextField()
    last_success_start = models.DateTimeField()


@transaction.atomic
def unlink_telegrams(usernames_to_delete: Iterable[str]):
    telegrams_to_delete = TelegramAccount.objects.filter(username__in=usernames_to_delete)
    telegrams_to_delete.update(user=None)

    memberships_ids = (
        TgMembership.objects
        .filter(account__in=telegrams_to_delete, is_active=True)
        .values_list('id', flat=True)
    )
    actions = [Action(action=ACTION.USER_DELETED_TELEGRAM, membership_id=pk) for pk in memberships_ids]
    Action.objects.bulk_create(actions)


class User(models.Model):
    username = models.TextField(unique=True, null=True)
    email = models.TextField(null=True)
    organization_id = models.IntegerField(null=True)
    affiliation = models.TextField(null=True)
    is_active = models.BooleanField(null=False, default=False)
    quit_at = models.DateField(null=True)
    join_at = models.DateField(null=True)
    leave_at = models.DateTimeField(null=True)

    def update_usernames(self, actual_telegram_usernames, is_new_user=False):
        """Привязывает указанные телеграм-аккаунты к этому сотруднику"""
        new_usernames = {username.lower() for username in actual_telegram_usernames}

        if is_new_user:
            old_usernames = set()
        else:
            if hasattr(self, 'filtered_accounts'):
                old_usernames = {x.username for x in self.filtered_accounts}
            else:
                old_usernames = set(
                    self.telegram_accounts
                    .filter(username__isnull=False, created_with_uhura=False)
                    .values_list('username', flat=True)
                )

        to_link = new_usernames - old_usernames
        to_unlink = old_usernames - new_usernames

        if to_unlink:
            unlink_telegrams(to_unlink)

        if to_link and self.is_active:  # К уволенному сотруднику телеграм-аккаунты не привязываем
            for username in to_link:
                TelegramAccount.objects.update_or_create(username=username, defaults={'user': self})

    def get_usernames(self):
        return [tglogin.username for tglogin in self.telegram_accounts.all()]

    def __str__(self):
        return f'{self.username}'


class TelegramAccount(models.Model):
    created_with_uhura = models.BooleanField(default=False)
    is_bot = models.BooleanField(default=False)
    is_tasha = models.BooleanField(default=False)
    user = models.ForeignKey(User, on_delete=models.CASCADE, related_name='telegram_accounts', null=True)
    username = models.TextField(db_index=True, unique=True, null=True)
    telegram_id = models.BigIntegerField(db_index=True, null=True, unique=True)

    def __str__(self):
        return f'{self.username}, {self.telegram_id}'


class TgGroupInfo(models.Model):
    class Meta:
        app_label = 'tasha'
        db_table = 'tasha_tggroupinfo'

    title = models.TextField()
    telegram_id = models.BigIntegerField(null=False, db_index=True, unique=True)
    deactivated = models.BooleanField(null=False, default=False, db_index=True)
    count_from_api = models.IntegerField(null=True, blank=True, default=None)
    last_successful_scan = models.DateTimeField(null=True, blank=True, db_index=True)
    should_add_slack_migration_bot = models.BooleanField(null=False, default=False)
    slack_migration_bot_added = models.BooleanField(null=False, default=False)
    chat_type = models.CharField(choices=CHAT_TYPE.choices, max_length=10, null=True, default=None)
    god_bot_number = models.IntegerField(null=True, blank=True, default=None)
    invite_link = models.TextField(null=True)
    auto_approve = models.BooleanField(default=False, null=True)
    only_yandex = models.BooleanField(default=False, null=False)

    def __str__(self):
        return f'{self.title}, {self.telegram_id}'


class TgMembership(models.Model):
    class Meta:
        app_label = 'tasha'
        db_table = 'tasha_membership'
        unique_together = (('account', 'group'),)

    account = models.ForeignKey(TelegramAccount, on_delete=models.CASCADE)
    group = models.ForeignKey(TgGroupInfo, on_delete=models.CASCADE, related_name='memberships')
    is_active = models.BooleanField(default=True)
    is_admin = models.BooleanField(default=False)
    first_notified_from_email = models.DateTimeField(null=True, db_index=True, default=None)

    def __str__(self):
        return f'{self.account}'


class TgWhitelistEntry(models.Model):
    user_account = models.ForeignKey(TelegramAccount, on_delete=models.CASCADE, related_name='whitelist_groups')
    group = models.ForeignKey(TgGroupInfo, on_delete=models.CASCADE, related_name='whitelist_accounts')
    author = models.ForeignKey(User, on_delete=models.SET_NULL, null=True, related_name='whitelist_invites')
    created_at = models.DateTimeField(auto_now_add=True)
    is_active = models.BooleanField(default=True)

    class Meta:
        app_label = 'tasha'
        db_table = 'tasha_whitelist_entry'
        unique_together = (('user_account', 'group'),)

    def __str__(self):
        return f'{self.group}.{self.user_account}'


class PrivacyPolicySignupEntry(models.Model):
    user = models.ForeignKey(User, on_delete=models.SET_NULL, null=True, related_name='signups')
    login = models.TextField()
    version = models.TextField(null=False)
    organization_id = models.IntegerField(null=False)
    signed_up_at = models.DateTimeField(auto_now_add=True)
    created_at = models.DateTimeField(auto_now_add=True)
    is_active = models.BooleanField(default=True)

    class Meta:
        app_label = 'tasha'
        db_table = 'tasha_privacy_policy_signup_entry'
        unique_together = (('user', 'organization_id', 'version'),)

    def __str__(self):
        return f'{self.signed_up_at}.{self.user}.{self.organization_id}.{self.version}'


class Action(models.Model):
    class Meta:
        app_label = 'tasha'
        db_table = 'tasha_actions'
        ordering = ('-added', '-id')

    added = models.DateTimeField(auto_now_add=True, db_index=True)
    action = models.TextField(choices=ACTION.choices)

    membership = models.ForeignKey(TgMembership, on_delete=models.CASCADE, null=True, default=None, related_name='actions')
    group = models.ForeignKey(TgGroupInfo, on_delete=models.CASCADE, null=True, default=None, related_name='actions')

    def __str__(self):
        return f'{self.action}, added at {self.added}'


class TelethonSession(models.Model):
    class Meta:
        app_label = 'tasha'
        db_table = 'tasha_telethonsession'

    session_name = models.CharField(max_length=32, null=False, blank=False, primary_key=True)
    dc_id = models.IntegerField(null=False)
    server_address = models.CharField(max_length=255, null=False, blank=False)
    port = models.IntegerField(null=False)
    auth_key = models.BinaryField(null=True)

    def __str__(self):
        return f'{self.session_name}'


class TelethonEntity(models.Model):
    class Meta:
        app_label = 'tasha'
        db_table = 'tasha_telethonentity'
        unique_together = (('session_name', 'telegram_id'),)

    session_name = models.CharField(max_length=32, null=False, blank=False, db_index=True)
    telegram_id = models.BigIntegerField(null=False, db_index=True)
    access_hash = models.BigIntegerField(null=False)
    username = models.CharField(max_length=255, null=True)
    phone = models.BigIntegerField(null=True)
    name = models.CharField(max_length=255, null=True)

    def __str__(self):
        return f'{self.session_name}'


class TelethonSentFile(models.Model):
    class Meta:
        app_label = 'tasha'
        db_table = 'tasha_telethonsentfile'
        unique_together = (('md5_digest', 'file_size', 'file_type'),)

    session_name = models.CharField(max_length=32, null=False, blank=False, db_index=True)
    md5_digest = models.BinaryField(db_index=True)
    file_size = models.IntegerField(null=False)
    file_type = models.IntegerField(null=False, db_index=True)
    file_id = models.BigIntegerField(null=False)
    access_hash = models.IntegerField(null=False)

    def __str__(self):
        return f'{self.session_name}'


class TgMessage(models.Model):
    added = models.DateTimeField(auto_now_add=True, db_index=True)
    user_telegram_id = models.BigIntegerField(null=False,  db_index=True)
    group_telegram_id = models.BigIntegerField(null=False, db_index=True)
