from collections import defaultdict
from operator import itemgetter
import statface_client
import datetime

from django.conf import settings
from django.db.models import Q, F, Count
from django.utils import timezone
from django.core.cache import cache

from tasha.constants import ACTION
from tasha.models import TgMembership, Action, TelegramAccount


class BaseMetric:
    report_name = None
    scale = 'minutely'
    fielddate_format = '%F %T'

    def calculate(self, *args, **kwrags):
        raise NotImplementedError

    def push_to_statface(self, *args, **kwrags):
        report_path = settings.STATFACE_REPORT_PATH.get(self.report_name)
        assert report_path is not None, 'Metric not have report_path'
        statface = statface_client.StatfaceClient(
            host=settings.STATFACE_HOST,
            oauth_token=settings.STATFACE_OAUTH_TOKEN,
        )
        remote_report = statface.get_report(report_path)

        fielddate = timezone.now().replace(second=0, microsecond=0)
        key_name = fielddate.strftime(self.fielddate_format)
        data = {'fielddate': key_name}
        for slug, value in self.calculate(*args, **kwrags):
            data[slug] = value
        remote_report.upload_data(scale=self.scale, data=[data])


class MembershipsCountInconsistencies(BaseMetric):
    report_name = 'memberships_count_inconsistencies'

    def calculate(self, *args, **kwargs):
        count = (
            TgMembership.objects.filter(group__deactivated=False)
            .exclude(group__count_from_api=None)
            .values('group_id', 'group__count_from_api')
            .annotate(count_from_db=Count('group_id'))
            .exclude(group__count_from_api=F('count_from_db')).count()
        )
        return [('inconsistent_chats', count)]


class DismissedNotKicked(BaseMetric):
    report_name = 'dismissed_not_kicked'

    def calculate(self, *args, **kwrags):
        report_date = timezone.now() - timezone.timedelta(days=1)
        not_kicked = (
            TgMembership
            .objects
            .filter(
                group__deactivated=False,
                is_active=True,
                account__is_bot=False,
                account__user__is_active=False,
                account__user__leave_at__lte=report_date,
            )
            .select_related('account__user')
            .distinct()
        )

        count = 0
        for membership in not_kicked:
            if not membership.actions.filter(added__gt=membership.account.user.leave_at, action='email').exists():
                count += 1
        return [('not_kicked', count)]


class TimeToKickFromChats(BaseMetric):
    report_name = 'time_to_kick'

    @staticmethod
    def _get_actions_about_kicks(report_date):
        # Выберем все actions начиная с начала даты отчета и у которых заполнено membership
        query = Q(added__gte=report_date, membership__isnull=False)
        # Выберем actions об удалении из группы или отправке письма, если первое письмо отправлено не раньше даты отчета
        query &= (
            Q(action__in=ACTION.user_kicked | {ACTION.USER_NOT_PARTICIPANT}) |
            Q(action=ACTION.EMAIL, membership__first_notified_from_email__gte=report_date)
        )

        return Action.objects.filter(query).values_list('id', flat=True)

    def get_kicks_or_fail_memberships(self, report_date):
        actions = self._get_actions_about_kicks(report_date)

        # Выберем те membership, у которых есть action об удалении из группы
        query = Q(tgmembership__actions__in=actions)
        # Или те, которые сейчас активны
        query |= Q(
            tgmembership__group__deactivated=False,
            tgmembership__first_notified_from_email=None,
            tgmembership__is_active=True,
        )
        # Выберем только те из них, где пользователь уволен или не указан (TASHA-29)
        query &= (Q(user__is_active=False) | Q(user__isnull=True))

        return (
            TelegramAccount
            .objects
            .filter(is_bot=False)
            .filter(query)
            .values_list('tgmembership', 'tgmembership__actions__added', 'user__leave_at', 'tgmembership__is_active')
        )

    def get_enter_actions(self, memberships):
        # Для membership из списка
        query = Q(membership_id__in=memberships)
        # Получим те, о которых есть информация о входе в чат
        query &= Q(action__in=ACTION.user_enter_to_chat | {ACTION.USER_CHANGED_USERNAME})
        # И в которых user уволен или не указан
        query &= (Q(membership__account__user__is_active=False) | Q(membership__account__user__isnull=True))

        return (
            Action
            .objects
            .filter(query)
            .order_by('added')
            .values_list('membership', 'added', 'action')
        )

    def _get_membership_kicked_time_delta(self, time_enters, time_exits):
        if time_exits is None:
            return [(timezone.now() - _enter).total_seconds() for _enter in time_enters]

        result = []
        i = 0
        for j in range(len(time_enters)):
            _enter = time_enters[j]
            # Найдем ближайшую дату кика из чата и запишем ее в результат
            for i in range(i, len(time_exits)):
                _exit = time_exits[i]
                if _exit > _enter:
                    result.append((_exit - _enter).total_seconds())
                    break
            # Если такого не нашлось,
            # то для всех оставшихся входов добавим в результат разницу между входом и текущей датой
            else:
                result += [(timezone.now() - _enter).total_seconds() for _enter in time_enters[j:]]
                break
        return result

    def get_time_delta_kicked(self, with_memberships=False):
        now = timezone.now()
        report_date = now - timezone.timedelta(days=30)
        memberships_to_metric = self.get_kicks_or_fail_memberships(report_date)

        kicks = defaultdict(set)
        enters = defaultdict(list)
        leaves = {}
        actives = {}
        for membership in memberships_to_metric:
            pk, kick, leave_at, is_active = membership
            leaves[pk] = leave_at
            actives[pk] = is_active
            kicks[pk].add(kick if kick is not None else now)
            if pk not in enters and leave_at and leave_at > timezone.datetime(2019, 5, 1, 0, 0, 0, 0, now.tzinfo):
                enters[pk].append((leave_at, None))

        membership_enters = self.get_enter_actions(kicks.keys())

        for membership in membership_enters:
            pk, join_at, action = membership
            if pk not in enters or join_at > enters[pk][0][0]:
                enters[pk].append((join_at, action))

        result = []
        for membership, joins in enters.items():
            kicks_ = sorted(kicks.get(membership, []))
            joins_ = []
            for join_at, action in sorted(joins, key=itemgetter(0), reverse=True):
                joins_.append(join_at)
                if action in ACTION.user_invalidations:
                    break

            joins_.reverse()
            if not actives[membership] and joins_ and leaves.get(membership) == joins_[-1] and kicks_ and kicks_[-1] < joins_[-1]:
                joins_.pop()

            membership_result = self._get_membership_kicked_time_delta(
                time_enters=joins_,
                time_exits=kicks_
            )

            if with_memberships:
                membership_result = add_membership(membership, membership_result)
            result += membership_result

        return result

    def calculate(self, *args, percentile=90, **kwargs):
        time_to_kick = self.get_time_delta_kicked()
        index = int(len(time_to_kick) / 100 * percentile)
        # В отчет отдаем время в часах
        time_in_hours = round(sorted(time_to_kick)[index-1] / 60 / 60, 2)
        return [('percentile_90', time_in_hours)]


def add_membership(membership, membership_result):
    return [(membership, x) for x in membership_result]


class TelegramStats(BaseMetric):
    scale = 'daily'
    fielddate_format = '%F'
    report_name = 'telegram_stats'

    def calculate(self, *args, **kwargs):
        today = datetime.date.today()
        today_str = today.isoformat()
        isocalendar = today.isocalendar()
        week_str = f'{isocalendar[0]}-{isocalendar[1]}'

        fieldname_to_cache_key = [
            ('messages_week', f'stat-messages-week:{week_str}'),
            ('messages_day', f'stat-messages-day:{today_str}'),
            ('wau', f'stat-wau:{week_str}'),
            ('dau', f'stat-dau:{today_str}'),
            ('chats_week', f'stat-chats-week:{week_str}'),
            ('chats_day', f'stat-chats-day:{today_str}'),
        ]
        result = []

        for fieldname, cache_key in fieldname_to_cache_key:
            if fieldname in ('wau', 'dau', 'chats_week', 'chats_day'):
                cache_value = cache.get(cache_key, [])
                value = len(cache_value)
            else:
                cache_value = cache.get(cache_key, -1)
                value = cache_value
            result.append((fieldname, value))
        return result
