#!/usr/bin/env python

from tasha import environment

environment.setup_environment()  # noqa: E402

import collections  # noqa: E402
import logging  # noqa: E402
import re  # noqa: E402
import requests  # noqa: E402
import time  # noqa: E402
import urllib3  # noqa: E402

from django.conf import settings  # noqa: E402
from slackclient import SlackClient  # noqa: E402

from tasha import models  # noqa: E402
from tasha.external import staff  # noqa: E402
from tasha.lib import mail  # noqa: E402

logger = logging.getLogger(__name__)
urllib3.disable_warnings()


class SlackUserInfo(object):
    def __init__(self, uid, updated, profile={}, teams=[], groups=[], channels=[], deleted=False, is_bot=False,
                 team_oauth_token=None, slack_client=None):
        self.uid = uid
        self.updated = updated
        self.profile = profile
        self.teams = teams
        self.groups = groups
        self.profile = profile
        self.channels = channels
        self.is_bot = is_bot
        self.team_oauth_token = team_oauth_token
        self.slack_client = slack_client

    def __str__(self):
        return '%s %s' % (self.uid, self.profile.get('email', ''))


class SlackGatekeeper(object):
    CHK_EMAIL = 0
    CHK_DISMISSED = 1
    CHK_DOMAIN_RESTR = 2
    CHK_BOT = 3
    CHK_UNABLE_TO_KICK = 4

    CRUMB_RE = re.compile(r'<input type="hidden" name="crumb" value="(.*)" />')
    TOKEN_RE = re.compile(r"api_token:\s'(.*)'")

    ALERT_TYPES = {
        CHK_EMAIL: '[SlackBot] User has invalid email (not yandex-team/yamoney)',
        CHK_DISMISSED: '[SlackBot] Dismissed user in a channel',
        CHK_DOMAIN_RESTR: '[SlackBot] Team has no email domain restrictions or restrictions are invalid',
        CHK_BOT: '[SlackBot] External Bot in a channel',
        CHK_UNABLE_TO_KICK: '[SlackBot] Unable to kick user'
    }

    def __init__(self):
        self.team_admins = collections.defaultdict(list)

    def kick_user(self, user):
        known_chans = []
        known_groups = []
        sc = user.slack_client
        token = user.team_oauth_token

        email = user.profile.get('email', '')
        if email in settings.WHITELIST:
            return

        logger.info('Try to kick %s', user)
        for ch in user.channels:
            chan_id = ch.get('id')
            if not chan_id or chan_id in known_chans:
                continue
            res = sc.api_call("channels.kick", user=user.uid, channel=chan_id)
            known_chans.append(chan_id)

        for group in user.groups:
            group_id = group.get('id')
            if not group_id or group_id in known_groups:
                continue
            res = sc.api_call("groups.kick", user=user.uid, channel=group_id)
            known_groups.append(group_id)

        # in paid version we can just run this call
        #        res = sc.api_call("users.admin.setInactive", user=user, set_active=True)
        for t in set(user.teams):
            try:
                res = requests.post(
                    'https://{}.slack.com/api/users.admin.setInactive'.format(t),
                    data=dict(user=user.uid, set_active=True, token=token),
                    headers={'User-Agent': settings.ADMIN_USER_AGENT}
                )
                resp_data = res.json()
            except Exception:
                pass
            else:
                if not resp_data.get('ok'):
                    if self.team_admins.get(t):
                        admins = list(filter(
                            lambda x: x != user.profile.get('email', ''),
                            self.team_admins.get(t, [])
                        ))
                        self.send_alert(
                            admins,
                            SlackGatekeeper.CHK_UNABLE_TO_KICK,
                            (t,),
                            user.profile.get('email', ''),
                        )
                    else:
                        logger.warning('No team admins: %r', (t, user, token, res.text))
                        self.send_alert(
                            None,
                            SlackGatekeeper.CHK_UNABLE_TO_KICK,
                            (t,),
                            user.profile.get('email', ''),
                        )
                else:
                    logger.info('Autokick successful: %s, %s, %s', t, user, res.text)
            logger.info('Team handled: %r', (t, user, token, res.text))

    def is_valid_email(self, email):
        if '@' in email:
            (_, domain) = email.split('@', 1)
            return domain in settings.VALID_DOMAINS
        return False

    def get_ya_username(self, email):
        login = email.split('@', 1)[0]
        if '+' in login:
            login = login.split('+')[0]
        return login

    def run(self):
        commands = 0
        user_cache = dict()
        tokens_qs = models.OAuthToken.objects.values_list('token', flat=True)

        for token in tokens_qs:
            sc = SlackClient(token)
            res = sc.api_call('team.info')
            if not res.get('ok'):
                continue
            commands += 1

            team_info = res.get('team')
            team_domain = team_info.get('domain')

            res = sc.api_call("users.list")
            if not res.get('ok'):
                continue

            for user in res.get('members', []):
                email = user.get('profile', {}).get('email')
                if user.get('is_admin') and email:
                    self.team_admins[team_domain].append(email)
                if user_cache.get(user.get('id')):
                    if user.get('deleted', False):
                        continue
                    email = user_cache[user.get('id')].profile.get('email', '')
                    user_cache[user.get('id')].teams.append(team_domain)
                    user_cache[user.get('id')].updated = time.time()
                else:
                    if user.get('deleted', False):
                        continue
                    user_cache[user.get('id')] = SlackUserInfo(
                        uid=user.get('id'),
                        profile=user.get('profile', {}),
                        groups=[],
                        channels=[],
                        updated=time.time(),
                        teams=[team_domain],
                        is_bot=user.get('is_bot', False),
                        team_oauth_token=token,
                        slack_client=sc,
                    )

        for uid, user in user_cache.items():
            # XXX: temporary hack
            if user.is_bot:
                continue
            channels = user.channels
            groups = user.groups
            teams = list(set(user.teams))
            email = user.profile.get('email', '')
            if not email:
                name = user.profile.get('real_name_normalized', '')
                self.send_alert(None, SlackGatekeeper.CHK_BOT, list(set(teams)), name, channels, groups)
                continue
            if email in settings.WHITELIST:
                continue

            if not self.is_valid_email(email):
                actual_teams = set()
                for t in teams:
                    actual_teams.add(t)
                if not actual_teams:
                    continue
                self.send_alert(None, SlackGatekeeper.CHK_EMAIL, list(actual_teams), email, channels, groups)
                self.kick_user(user)
                continue

            username = self.get_ya_username(email)
            if staff.is_dismissed_person(username):
                logger.info('Dismissed user %r', user.profile)
                actual_teams = set()
                for t in teams:
                    actual_teams.add(t)
                if not actual_teams:
                    continue
                self.send_alert(None, SlackGatekeeper.CHK_DISMISSED, list(actual_teams),
                                email, channels, groups, is_dismissed=True)
                self.kick_user(user)
                continue
        return commands

    def send_alert(self, to, alert_type, teams, email, channels=[], groups=[], is_dismissed=False):
        if not email or not teams:
            return
        if not to:
            to = ['slack-security@yandex-team.ru']

        subject = self.ALERT_TYPES.get(alert_type)
        context = dict(
            alert_type=alert_type, teams=teams, channels=channels,
            groups=groups, email=email, is_dismissed=is_dismissed
        )
        mail.send_mail(to, subject, 'slack_notify', context)


def main():
    run_start = time.time()
    b = SlackGatekeeper()
    commands = b.run()
    statistics = {
        'check_time': time.time() - run_start,
        'total_teams': commands,
    }
    models.SlackStatistics(statistics=statistics).save()


if __name__ == '__main__':
    try:
        main()
    except Exception as err:
        logger.exception('Uncaught exception %s: %s', err.__class__.__name__, err)
