# coding: utf-8

import logging

from django import http
from django import shortcuts
from django.conf import settings
from django.views.generic import base
import requests

from tasha import models


logger = logging.getLogger(__name__)


class SlackCallbackView(base.View):
    def get(self, request):
        code = request.GET.get('code')
        if not code:
            return http.HttpResponseBadRequest('parameter "code required"')
        oauth_token = self._slack_oauth_access_request(code)
        token_obj = models.OAuthToken(token=oauth_token)
        token_obj.save()
        return shortcuts.render_to_response('slack_ok.html')

    def _slack_oauth_access_request(self, code, retries=3):
        data = {
            'client_id': settings.SLACK_OAUTH_CLIENT_ID,
            'client_secret': settings.SLACK_OAUTH_CLIENT_SECRET,
            'code': code,
            'redirect_uri': settings.SLACK_OAUTH_REDIRECT_URI,
        }
        try:
            response = requests.post(
                settings.SLACK_OAUTH_ACCESS_URL,
                data=data,
            )
            response.raise_for_status()
            r = response.json()
            return r['access_token']
        except (requests.exceptions.ConnectTimeout, requests.exceptions.ReadTimeout):
            if retries > 0:
                return self._slack_oauth_access_request(code, retries=retries - 1)
            else:
                logger.exception('Slack request timeout exception')
        except requests.exceptions.HTTPError:
            if response and response.status_code == 404:
                logger.error('Url not found: %s', response.url)
            elif retries > 0:
                return self._slack_oauth_access_request(code, retries=retries - 1)
            else:
                logger.error(
                    'Could not access URL %s, received HTTP %d %s',
                    response.url,
                    response.status_code,
                    response.reason,
                    exc_info=True
                )
        except (ValueError, KeyError):
            logger.error('Bad json response URL %s, body %s', response.url, response.text)
            raise
