# coding: utf-8
import collections
import logging
import json
import os
import time
from collections import Counter
from typing import Optional

from django import http
from django.conf import settings
from django.db import DatabaseError
from django.db.models import Q, F, Count
from django.http import HttpResponseForbidden, HttpResponseBadRequest, JsonResponse
from django.shortcuts import get_object_or_404
from django.views.generic import base
from django.utils import timezone
from django_pgaas import atomic_retry
import sqlite3

from telethon.errors import InviteHashExpiredError, UserPrivacyRestrictedError
from trafaret.constructor import construct
from trafaret import DataError

from tasha.external.telethon import api_sync as telethon_api_sync
from tasha.lib import telegram, cache_storage
from tasha.models import TgMembership, TelegramAccount, TgGroupInfo, TgMessage
from tasha.monitorings.metrics import TimeToKickFromChats
from tasha.views.forms import SendMessageForm, AddSlackBotForm

logger = logging.getLogger(__name__)


def validate_and_decode_internal_telegram_request(request):
    host = request.get_host().split(':')[0]
    logger.debug('request host %s', host)
    if not any([host.endswith(x) for x in settings.INTERNAL_ALLOWED_HOSTS]):
        return HttpResponseForbidden('Request host is not in allowed hosts'), True
    try:
        json_body = json.loads(request.body.decode('utf-8'))
        return json_body, False
    except ValueError:
        return HttpResponseBadRequest('Invalid json in request.body: %r' % request.body), True


@atomic_retry
def save_chat_to_db(chat_obj, has_true_bot):
    if not isinstance(chat_obj, dict):
        chat_obj = chat_obj.to_dict()
    deactivated = chat_obj.get('deactivated', False)
    client = telethon_api_sync.get_client_sync()
    client.session.save_entities = False
    bot = client.get_me()
    bot_username = TelegramAccount.objects.get(telegram_id=bot.id)
    true_bot_record = TelegramAccount.objects.get(username=settings.TELEGRAM_TRUE_BOT_USERNAME)

    telegram_id = chat_obj['id']
    if chat_obj.get('megagroup', False) and telegram_id > 0:
        # convert chat_id to supergroup id
        telegram_id = int('-100' + str(telegram_id))

    group, _ = TgGroupInfo.objects.get_or_create(telegram_id=telegram_id, deactivated=deactivated, title=chat_obj['title'])
    TgMembership.objects.get_or_create(group=group, account=bot_username)
    if has_true_bot:
        TgMembership.objects.update_or_create(group=group, account=true_bot_record, defaults={'is_active': True})


def get_group_by_telegram_id_or_title(telegram_id: Optional[int], title: Optional[str]) -> TgGroupInfo:
    assert (telegram_id is None) != (title is None)

    qs = TgGroupInfo.objects.filter(deactivated=False)

    if telegram_id:
        group = get_object_or_404(qs, telegram_id=telegram_id)

    else:
        try:
            group = get_object_or_404(qs, title=title)
        except TgGroupInfo.MultipleObjectsReturned:
            group = get_object_or_404(qs, title=title, telegram_id__lt=0)

    return group


class ChatParticipantsView(base.View):
    def get(self, request):
        title = request.GET.get('title')
        chat_id = request.GET.get('chat_id')
        if title is None and chat_id is None:
            return http.HttpResponseBadRequest('`title` or `chat_id` must be provided')

        try:
            group = get_group_by_telegram_id_or_title(chat_id, title)
        except TgGroupInfo.MultipleObjectsReturned:
            return http.JsonResponse({'error': f'multiple groups with title `{title}` exist'}, status=400)

        result = list(
            group.memberships
            .filter(account__user__is_active=True, is_active=True)
            .annotate(login=F('account__user__username'))
            .values('login', 'is_admin')
        )
        return http.JsonResponse(result, safe=False)


class ExportChats(base.View):
    def get(self, request):
        chats = (
            TgGroupInfo.objects.filter(deactivated=False).values(
                'title',
                'telegram_id',
                'count_from_api',
                'id',
                'should_add_slack_migration_bot',
                'slack_migration_bot_added',
            )
        )
        stats = dict(
            TgMessage.objects.values('group_telegram_id').filter(
                added__gt=timezone.now() - timezone.timedelta(days=10)
            ).annotate(x=Count('group_telegram_id')).values_list('group_telegram_id', 'x')
        )
        memberships = (
            TgMembership.objects.filter(
                account__user__is_active=True, is_active=True
            ).values('account__telegram_id', 'account__username', 'is_admin', 'group__telegram_id')
        )
        chat_to_members = collections.defaultdict(list)
        for membership in memberships:
            chat_to_members[membership['group__telegram_id']].append({
                'is_admin': membership['is_admin'],
                'user_telegram_id': membership['account__telegram_id'],
                'user_telegram_username': membership['account__username']
            })

        result = []
        for chat in chats:
            result.append({
                'title': chat['title'],
                'telegram_id': chat['telegram_id'],
                'should_add_slack_migration_bot': chat['should_add_slack_migration_bot'],
                'slack_migration_bot_added': chat['slack_migration_bot_added'],
                'user_count_from_api': chat['count_from_api'],
                'message_count': stats.get(chat['telegram_id'], 0),
                'memberships': chat_to_members.get(chat['telegram_id'], []),
            })
        result.sort(key=lambda item: item['message_count'] or 0, reverse=True)
        return http.JsonResponse({'data': result})


class TgGroupInfoView(base.View):
    def get(self, request):
        title = request.GET.get('title')
        chat_id = request.GET.get('chat_id')
        if title is None and chat_id is None:
            return http.HttpResponseBadRequest('`title` or `chat_id` must be provided')

        try:
            group = get_group_by_telegram_id_or_title(chat_id, title)
        except TgGroupInfo.MultipleObjectsReturned:
            return http.JsonResponse({'error': f'multiple groups with title `{title}` exist'}, status=400)

        return http.JsonResponse({
            "title": group.title,
            "telegram_id": group.telegram_id,
            'bots': list(
                group.memberships.filter(is_active=True, account__is_tasha=True).values_list('account__username', flat=True)
            )
        })


class KnownChatView(base.View):
    def get(self, request):
        telegram_id = request.GET.get('telegram_id')
        if telegram_id is None:
            return http.HttpResponseBadRequest('`telegram_id` must be provided')

        if TgGroupInfo.objects.filter(deactivated=False, telegram_id=telegram_id).exists():
            return http.JsonResponse({'ok': True})
        else:
            return http.JsonResponse({'ok': False})


class InviteQBotToChat(base.View):
    def post(self, request):
        telegram_id = int(request.GET.get('telegram_id'))
        client = telethon_api_sync.get_client_sync()
        chat_obj = client.get_entity(telegram_id)
        telethon_api_sync.invite_user_to_group(
            chat_obj,
            username=settings.YAMBOGRAM_USERNAME,
            is_bot=True,
        )

        return JsonResponse({'status': 'ok'})


class InviteToChatView(base.View):
    def post(self, request):
        json_body, error = validate_and_decode_internal_telegram_request(request)
        if error:
            return json_body
        invite_link = json_body.get('invite_link')
        if invite_link is None:
            return HttpResponseBadRequest('Parameter "invite_link" is required')
        try:
            if telethon_api_sync.check_already_participants(invite_link):
                status, chat_obj = 'already_participant', None
            else:
                status, chat_obj = telethon_api_sync.join_chat(invite_link)
        except InviteHashExpiredError:
            status, chat_obj = 'expired_link_or_banned', None
        except Exception:
            logger.exception('Invite to chat exception occured')
            raise

        if chat_obj:
            # добавляем бота-бота вслед за нами
            has_true_bot = False
            try:
                telethon_api_sync.invite_user_to_group(
                    chat_obj,
                    username=settings.TELEGRAM_TRUE_BOT_USERNAME,
                    make_admin=True,
                    is_bot=True,
                    announce_message=telegram.ADD_BOT_ANNOUNCE_MESSAGE.format(
                        bot_username=settings.TELEGRAM_TRUE_BOT_USERNAME
                    ),
                )
                has_true_bot = True
            except Exception:
                logger.exception('Successfully added semibot to chat %s, but failed to add true bot', chat_obj.id)
            try:
                save_chat_to_db(chat_obj, has_true_bot)
            except (DatabaseError, sqlite3.DatabaseError):
                pass  # не страшно, потом досинкаем (наверное)

        return JsonResponse({'status': status})


class CreateChatView(base.View):

    __request_validator = construct({'telegram_id': int, 'title': str, 'telegram_username': str})

    def post(self, request):
        (json_body, error) = validate_and_decode_internal_telegram_request(request)
        if error:
            return json_body
        try:
            self.__request_validator(json_body)
        except DataError as error:
            logger.error('Incorrect request: %s', error)
            return http.HttpResponseBadRequest(json.dumps(error.as_dict()))
        telegram_id = int(json_body['telegram_id'])
        title = json_body['title']
        telegram_username = json_body['telegram_username']

        if not telegram.validate_telegram_id(telegram_id):
            return http.HttpResponseForbidden('Unknown telegram_id for UhuraToolsBot')
        try:
            chat_obj = telethon_api_sync.create_supergroup(title)
            user_phones = telegram.get_user_phones_by_telegram_id(telegram_id)
            telethon_api_sync.invite_user_to_group(
                chat_obj,
                user_id=telegram_id,
                username=telegram_username,
                phones=user_phones,
                make_admin=True,
            )
            # добавляем бота-бота вслед за нами
            has_true_bot = False
            try:
                telethon_api_sync.invite_user_to_group(
                    chat_obj,
                    username=settings.TELEGRAM_TRUE_BOT_USERNAME,
                    make_admin=True,
                    is_bot=True,
                    announce_message=telegram.ADD_BOT_ANNOUNCE_MESSAGE.format(
                        bot_username=settings.TELEGRAM_TRUE_BOT_USERNAME
                    ),
                )
                has_true_bot = True
            except Exception:
                logger.exception('Successfully created chat %s, but failed to add true bot', chat_obj.id)
            try:
                save_chat_to_db(chat_obj, has_true_bot)
            except (DatabaseError, sqlite3.DatabaseError):
                pass  # не страшно, потом досинкаем (наверное)
        except UserPrivacyRestrictedError as error:
            logger.error('Create chat exception occured: %s', error)
            return http.JsonResponse({'status': 'privacy_error'})
        except Exception:
            logger.exception('Create chat exception occured')
            raise
        else:
            return http.JsonResponse({'status': 'success'})


class KickDeltasMonitoringView(base.View):
    SECONDS_IN_DAY = 60 * 60 * 24

    def get(self, request):
        kick_deltas = self.get_kick_deltas()
        if not kick_deltas:
            return http.HttpResponse(status=204)

        return http.HttpResponse(
            '\n'.join(
                f'Membership {membership} not kicked for {x} seconds'
                for membership, x in kick_deltas
            ),
            status=500,
        )

    def get_kick_deltas(self):
        kick_deltas1 = set(TimeToKickFromChats().get_time_delta_kicked(with_memberships=True))
        kick_deltas2 = set(TimeToKickFromChats().get_time_delta_kicked(with_memberships=True))
        kick_deltas = kick_deltas2 - kick_deltas1
        kick_deltas = [delta for delta in kick_deltas if delta[1] > self.SECONDS_IN_DAY]
        return kick_deltas


class NotScannedMonitoringView(base.View):
    def get(self, request):
        failed_groups = TgGroupInfo.objects.filter(deactivated=False).filter(
            Q(last_successful_scan__lt=timezone.now() - timezone.timedelta(days=2)) |
            Q(last_successful_scan=None)
        )
        if not failed_groups:
            return http.HttpResponse(status=204)

        return http.HttpResponse(
            'Failed groups: %s' % len(failed_groups),
            status=500,
        )


class HangingMonitoring(base.View):
    def get(self, request):
        fails = []
        bots_usernames = (
            TelegramAccount.objects
            .filter(is_tasha=True)
            .values_list('username', flat=True)
            .order_by('username')
        )
        for username in bots_usernames:
            last_finish = cache_storage.get_key_value(f'last_finish_{username}')
            if last_finish == cache_storage.EXPIRED_VALUE:
                fails.append(f'{username} is not in cache')
            elif time.time() - last_finish > settings.MAX_HANGING_TIME:
                fails.append(f'{username} is hanging')

        if not fails:
            return http.HttpResponse(status=204)

        return http.HttpResponse(
            '\n'.join(fails),
            status=500,
        )


class GroupsCountMonitoring(base.View):
    def get(self, request):
        counter = Counter(
            TgMembership.objects
            .filter(account__is_tasha=True, group__deactivated=False,)
            .values('account__username', 'group__title')
            .annotate(x=Count('group__title', distinct=True))
            .values_list('account__username', 'x')
        )
        if not any(k for k, v in counter.items() if v < settings.MAX_GROUPS_COUNT_FOR_MONITORING):
            return http.HttpResponse(status=500)

        return http.HttpResponse(status=204)


class SendMessageView(base.View):
    def post(self, request):
        form = SendMessageForm(json.loads(request.body))
        if not form.is_valid():
            return HttpResponseBadRequest(form.errors)
        chat_id = form.cleaned_data['chat_id']
        message = form.cleaned_data['message']

        client = telethon_api_sync.get_client_sync()
        chat_obj = client.get_entity(chat_id)

        telethon_api_sync.send_message(chat_obj, message)

        return http.JsonResponse({'ok': True})


class AddSlackBotFlagView(base.View):
    def post(self, request):
        form = AddSlackBotForm(json.loads(request.body))
        if not form.is_valid():
            return http.JsonResponse({'error': 'invalid chat id'}, status=400)
        chat_id = form.cleaned_data['chat_id']
        chats = TgGroupInfo.objects.filter(telegram_id=chat_id)
        if not chats.exists():
            return http.JsonResponse({'error': f'no chat with id {chat_id}'}, status=400)
        chats.update(should_add_slack_migration_bot=True)
        return http.JsonResponse({'ok': True})


class AddSlackBotView(base.View):
    def post(self, request):
        form = AddSlackBotForm(json.loads(request.body))
        if not form.is_valid():
            return http.JsonResponse({'error': 'invalid chat id'}, status=400)
        chat_id = form.cleaned_data['chat_id']
        chats = TgGroupInfo.objects.filter(telegram_id=chat_id)
        if not chats.exists():
            return http.JsonResponse({'error': f'no chat with id {chat_id}'}, status=400)
        if chats.count() > 1:
            return http.JsonResponse({'error': f'two or more chats with id {chat_id}'}, status=400)
        tg_info: TgGroupInfo = chats.get()

        slack_bot_username = os.environ.get('TELEGRAM_SLACK_BOT_USERNAME', 'slack_migration_helper_bot')
        client = telethon_api_sync.get_client_sync()
        chat_obj = client.get_entity(chat_id)
        telethon_api_sync.invite_user_to_group(
            chat_obj,
            username=slack_bot_username,
            make_admin=True,
            is_bot=True,
            announce_message=None,
        )
        tg_info.should_add_slack_migration_bot = True
        tg_info.slack_migration_bot_added = True
        tg_info.save()
        return http.JsonResponse({'ok': True})
