import json

from django import http
from django.core.handlers.wsgi import WSGIRequest
from django.db.models import Count
from django.http import HttpResponseBadRequest
from django.utils import timezone
from django.views.generic import base

from tasha.models import TgMembership, TelegramAccount, SyncTime
from tasha.views.forms import RegistrationForm


class BotsView(base.View):
    def get(self, request):
        latest_statistics = TgMembership.objects.filter(
            is_active=True,
            account__is_tasha=True,
            group__deactivated=False
        ).values('account__username').annotate(
            chats_count=Count('account__username')
        ).order_by('chats_count')
        bots = list(latest_statistics.values_list('account__username', flat=True))
        return http.JsonResponse(bots, safe=False)


class RegistrationView(base.View):
    def post(self, request: WSGIRequest):
        form = RegistrationForm(json.loads(request.body))
        if not form.is_valid():
            return HttpResponseBadRequest(form.errors)

        user = form.cleaned_data.get('user')
        telegram_id = form.cleaned_data.get('telegram_id')
        TelegramAccount.objects.update_or_create(
            telegram_id=telegram_id,
            defaults={
                'created_with_uhura': True,
                'username': None,
                'user': user,
            }
        )
        return http.JsonResponse({})


class StaffSyncMonitoring(base.View):
    SECONDS_IN_HOUR = 60 * 60

    def get(self, request):
        last_sync_hours = timezone.timedelta(hours=0)
        try:
            sync_time = SyncTime.objects.get(name='import_staff')
        except SyncTime.DoesNotExist:
            status = 500
        else:
            last_sync_hours = timezone.now() - sync_time.last_success_start
            status = 200 if last_sync_hours < timezone.timedelta(hours=1) else 500
        return http.HttpResponse(
            content='Last sync was finished %.1f hours ago' %
                    (last_sync_hours.total_seconds() / self.SECONDS_IN_HOUR),
            status=status
        )
