import logging

from tasha.core import TashaUnit
from tasha.core.services import UserService
from tasha.core.tasha_worker import job, retry_on_exception

logger = logging.getLogger(__name__)


@job()
async def kick_leaved_user(ctx, username: str) -> None:
    if not username:
        logger.error('no username provided')
        return

    async with ctx['db'].acquire_connection() as conn:
        async with TashaUnit(conn=conn, db_proxy=ctx['db'], bot=ctx['bot'], redis=ctx['redis']) as tu:
            user_srv = UserService(tu)
            user = await user_srv.get_user(username)
            if not user:
                logger.error('User not found for %s', username)
                return
            chats = await user_srv.get_chats_with_account(user)

            if not chats:
                logger.info('[%s] No Active chats found', username)
                return

            logger.info('[%s] found memberships in %s chats', username, len(chats))

            for chat in chats:
                await user_srv.kick_from_chat(
                    user=user,
                    chat_id=chat.group_telegram_id,
                    account_id=chat.account_id,
                    telegram_id=chat.account_telegram_id
                )


@job()
@retry_on_exception()
async def unban_user(ctx, username: str) -> None:
    if not username:
        logger.error('no username provided')
        return

    async with ctx['db'].acquire_connection() as conn:
        async with TashaUnit(conn=conn, db_proxy=ctx['db'], bot=ctx['bot'], redis=ctx['redis']) as tu:
            service = UserService(tu)
            user = await service.get_user(username)
            await service.unban_user(user)
