import asyncio
import logging

from tasha.config import settings
from tasha.core import TashaUnit
from tasha.core.services.tasha_service import TashaService
from tasha.core.tasha_worker import cron_job
from tasha.lib.export.slack import SlackExport
from tasha.lib.export.telegram import TelegramExport
from tasha.lib.export.chat_stat_export import ChatStatExport

logger = logging.getLogger(__name__)


@cron_job(
    weekday=5,
    timeout=10800,
)
async def export_tg_chats_stats(ctx) -> None:
    logger.info('Started export_tg_chats_stats')

    async with ctx['db'].acquire_connection() as conn:
        uow = TashaUnit(conn=conn, db_proxy=ctx['db'], bot=ctx['bot'], redis=ctx['redis'])
        async with uow:
            export = ChatStatExport(tu=uow)
            await export.export_chat_stats(ctx['db'])
    await asyncio.sleep(1)


@cron_job(
    hour={1},
    timeout=10800,  # 3h
)
async def export_telegram_chats(ctx) -> None:
    logger.info('Start export_telegram_chats')

    async with ctx['db'].acquire_connection() as conn:
        uow = TashaUnit(conn=conn, db_proxy=ctx['db'], bot=ctx['bot'], redis=ctx['redis'])
        async with uow:
            export = TelegramExport(tu=uow)
            # // TODO: заменить DBProxy на TashaUnit
            await export.export(ctx['db'])
    await asyncio.sleep(1)


@cron_job(
    hour={1},
    timeout=10800,  # 3h
)
async def export_slack_chats(ctx) -> None:
    logger.info('Start export_slack_chats')

    async with ctx['db'].acquire_connection() as conn:
        uow = TashaUnit(conn=conn, db_proxy=ctx['db'], bot=ctx['bot'], redis=ctx['redis'])
        async with uow:
            export = SlackExport(tu=uow)
            # // TODO: заменить DBProxy на TashaUnit
            await export.export(ctx['db'])
    await asyncio.sleep(1)


@cron_job(
    hour={1},
    run_at_startup=True,
)
async def create_unban_jobs(ctx) -> None:
    async with ctx['db'].acquire_connection() as conn:
        async with TashaUnit(conn=conn, db_proxy=ctx['db'], bot=ctx['bot'], redis=ctx['redis']) as tu:
            service = TashaService(tu=tu)
            usernames = await service.get_users_to_unban()
            logger.info('Found %s users to unban', len(usernames))
            for username in usernames:
                tu.add_job('unban_user', username=username, unique=True)


@cron_job(
    minute=set(range(0, 60, 5)),
)
async def create_chat_scan_jobs(ctx) -> None:
    async with ctx['db'].acquire_connection() as conn:
        async with TashaUnit(conn=conn, db_proxy=ctx['db'], bot=ctx['bot'], redis=ctx['redis']) as tu:
            service = TashaService(tu=tu)
            jobs = await service.get_queue_length(queue_name=settings.ARQ_SLOW_QUEUE_NAME)
            logger.info('get_queue_length %s', len(jobs))
            if len(jobs) < settings.ARQ_JOBS_COUNT_THRESHOLD:
                chat_ids = await service.get_least_scanned_chats(limit=500)
                logger.info('Will enquire %s chats to scan', len(chat_ids))
                for chat_id in chat_ids:
                    tu.add_slow_job('scan_chat', chat_id=chat_id)
