import aiohttp
import logging
from typing import Optional, Any

from tvm2 import TVM2
from fastapi import Request
from tvmauth import BlackboxTvmId as BlackboxClientId

from intranet.trip.ext_api.src.config import settings

logger = logging.getLogger(__name__)

FORWARDED_FOR_HEADER = 'X-Forwarded-For-Y'

SERVICE_TICKET_HEADER = 'X-Ya-Service-Ticket'
USER_TICKET_HEADER = 'X-Ya-User-Ticket'


def get_tvm_client():
    return TVM2(client_id=settings.TVM_CLIENT_ID, blackbox_client=BlackboxClientId.Prod)


async def get_tvm_service_ticket(destination: str) -> Optional[str]:
    destination = settings.TVM_SERVICES.get(destination)
    if not destination:
        return
    return await get_tvm_client().get_service_ticket(destination=str(destination))


def get_real_ip(headers: dict[str, str]) -> str:
    return headers.get(FORWARDED_FOR_HEADER)


def get_host(headers: dict[str, str]) -> str:
    for header in ('Host', 'HOST'):
        if header in headers:
            return headers[header]


def get_request_params(request: Request) -> dict[str, Any]:
    params = {
        'oauth_token': get_oauth_token(request) or '',
        'dbfields': '',
        'userip': get_real_ip(request.headers) or '',
        'host': get_host(request.headers),
        'emails': 'getdefault',
        'get_user_ticket': 'true',
        'regname': 'yes',
        'aliases': 'all',
        'method': 'oauth',
        'format': 'json',
    }
    return params


async def get_tvm_service_ticket_header(destination: str) -> dict[str, str]:
    service_ticket = await get_tvm_service_ticket(destination)
    return {
        SERVICE_TICKET_HEADER: service_ticket,
    }


async def get_user_by_request(request: Request) -> dict[str, Any]:
    """
    Возвращает dict с данными сотрудника от Блэкбокса.
    Интересующие нас поля:
    {
        'error': 'OK',
        'uid': {'value': '1120000000018264', 'lite': False, 'hosted': False},
        'login': 'wlame',
        'user_ticket': '3:user:CJFpEJfk9vcFGigKCQjYjpmRpdT-ARDYjpmRp.....',
        ...
    }
    """
    if settings.ENV_TYPE == 'development':
        return {
            'error': 'OK',
            'uid': {'value': '1120000000000000', 'lite': False, 'hosted': False},
            'login': 'dev',
            'user_ticket': 'user:ticket',
        }

    service_headers = await get_tvm_service_ticket_header('passport')
    params = get_request_params(request)

    async with aiohttp.ClientSession(headers=service_headers) as session:
        async with session.get(settings.BLACKBOX_URL, params=params) as response:
            response_data = await response.json()

    return response_data


def get_oauth_token(request: Request) -> str:
    """Example
    Authorization: OAuth vF9dft4qmT
    """
    headers = request.headers

    try:
        name, token = headers.get('Authorization', '').split(' ')
        if name.lower() == 'oauth':
            return token
    except ValueError:
        pass
    return None
