import uuid

from cached_property import cached_property
from starlette.routing import Match

from .base import BaseProvider, context_field


class RequestProvider(BaseProvider):
    @property
    def http_request(self):
        return self.kwargs.get('request')

    @cached_property
    def matching_route(self):
        if self.http_request is None:
            return None
        for route in self.http_request.scope['app'].routes:
            if route.matches(self.http_request)[0] == Match.FULL:
                return route
        return None

    @context_field('http')
    def user(self):
        user = self.http_request.state.user
        return {
            'login': user.login,
            'uid': user.uid,
        }

    @context_field('http')
    def request(self):
        return {
            'method': self.http_request.method,
            'path': self.http_request.url.path,
            'operation_id': getattr(self.matching_route, 'operation_id', ''),
            'tags': getattr(self.matching_route, 'tags', ''),
            'route': getattr(self.matching_route, 'path', ''),
        }

    @context_field('http')
    def request_id(self):
        return self.http_request.headers.get('X-Request-ID') or str(uuid.uuid4())

    @context_field('http')
    def client_session_id(self):
        return self.http_request.headers.get('X-Client-Session-ID')

    @context_field('http')
    def client_version(self):
        return self.http_request.headers.get('X-Client-Version')

    @context_field('http')
    def trip_id(self):
        return self.http_request.headers.get('X-Trip-ID')

    @context_field('http')
    def person_id(self):
        return self.http_request.headers.get('X-Person-ID')
