from collections import defaultdict
from cached_property import cached_property

from intranet.trip.src.config import settings
from intranet.trip.src.enums import Citizenship, PersonRole, PTStatus
from intranet.trip.src.exceptions import PermissionDenied
from intranet.trip.src.lib.utils import safe_getitem
from intranet.trip.src.models import (
    Trip,
    PersonTrip,
    Person,
    Service,
    Transaction,
    User,
)
from intranet.trip.src.unit_of_work import UnitOfWork


class LazyRolesMap:
    """
    Ленивое получение ролей пользователей по person_id
    """
    def __init__(self, uow: UnitOfWork, person_ids: list[int]):
        self._uow = uow
        self._person_ids = person_ids

    @cached_property
    async def relations_mapping(self) -> dict[int, dict[PersonRole, list[int]]]:
        items = await self._uow.person_relationships.get_relationships(
            roles=[PersonRole.chief, PersonRole.approver],
            person_ids=self._person_ids,
        )
        result = defaultdict(lambda: defaultdict(list))
        for item in items:
            result[item.dependant_id][item.role].append(item.owner_id)
        return result

    @cached_property
    async def managers_mapping(self) -> dict[int, list[int]]:
        return await self._uow.person_trips.get_manager_ids_for_persons(self._person_ids)

    async def _get_owners(self, dependant_id: int, role: PersonRole) -> list[int]:
        assert dependant_id in self._person_ids
        mapping = await self.relations_mapping
        return safe_getitem(mapping, [dependant_id, role], default=[])

    async def _has_role(self, owner_id: int, dependant_id: int, role: PersonRole) -> bool:
        return owner_id in await self._get_owners(dependant_id, role)

    async def is_chief(self, chief_id: int, person_id: int) -> bool:
        return await self._has_role(chief_id, person_id, PersonRole.chief)

    async def is_approver(self, approver_id: int, person_id: int) -> bool:
        return await self._has_role(approver_id, person_id, PersonRole.approver)

    async def is_manager(self, manager_id: int, person_id: int) -> bool:
        assert person_id in self._person_ids
        mapping = await self.managers_mapping
        return manager_id in mapping.get(person_id, [])

    async def get_approver_ids(self, person_id: int) -> list[int]:
        return await self._get_owners(person_id, PersonRole.approver)


class LazyManagersMapByTrip:
    """
    Ленивое получение менеджеров по идентификатору командировки.
    """
    def __init__(self, uow: UnitOfWork, trip_ids: list[int]):
        self._uow = uow
        self._trip_ids = trip_ids

    @cached_property
    async def mapping(self) -> dict[int, list[int]]:
        return await self._uow.person_trips.get_manager_ids(self._trip_ids)

    async def is_manager(self, person_id: int, trip_id: int) -> bool:
        assert trip_id in self._trip_ids
        mapping = await self.mapping
        return person_id in mapping.get(trip_id, [])


def is_yandex_coordinator(user: User) -> bool:
    """
    Checking that the user is the yandex coordinator or not

    :param user: the user whose perms are being checked
    :return: True if user is a yandex coordinator
    """
    return (
        user.is_coordinator
        and user.company.holding_id == settings.YANDEX_HOLDING_ID
    )


def is_holding_coordinator(user: User, holding_id: int) -> bool:
    """
    Checking that the user is the holding/yandex coordinator or not

    :param user: the user whose perms are being checked
    :param holding_id:
    :return: True if flag is_coordinator set and user.holding_id = holding_id,
    also return True if user is yandex coordinator
    """
    return (
        user.is_coordinator
        and user.company.holding_id in (holding_id, settings.YANDEX_HOLDING_ID)
    )


def is_support_for_person_trip(
    user: User,
    person_trip: PersonTrip,
):
    """
    Check permission for read pt, if user is support for person

    :param user:
    :param person_trip:
    :return:
    """
    if user.person_id != person_trip.person.support_id:
        return False
    if person_trip.status in (PTStatus.closed, PTStatus.cancelled):
        return False
    return True


async def has_manager_perm(
    user: User,
    person: Person,
    roles_map: LazyRolesMap,
) -> bool:
    """
    Only for users who are managers of one of the person's trip

    :param user: the user whose manager perms are being checked
    :param person: the person whose data the user reads
    :param roles_map: managers of persons trips by user.person_id
    :return:
    """
    return await roles_map.is_manager(user.person_id, person.person_id)


async def has_person_partial_perm(
    user: User,
    person: Person,
    roles_map: LazyRolesMap,
) -> bool:
    """
    All unlimited users of the holding and yandex coordinator have partial perms
    All users who are managers of one of the person's trip

    Using this ONLY when you need to get public data about person

    :param user: the user whose perms are being checked
    :param person: the person whose data the user reads
    :param roles_map: managers of persons trips by user.person_id
    :return:
    """
    if is_holding_coordinator(user, person.company.holding_id):
        return True
    if await has_manager_perm(user, person, roles_map):
        return True
    if user.person_id == person.support_id:
        return True
    return (
        user.company.holding_id == person.company.holding_id
        and (
            not user.is_limited_access
            or user.person_id == person.person_id
        )
    )


def has_person_perm(user: User, person: Person) -> bool:
    """
    Only the user, the holding/yandex coordinator and support have access perms
    This method is preferred when checking permissions for access to personal data

    :param user: the user whose perms are being checked
    :param person: the person whose data the user reads
    :return: user == person or is_holding_coordinator
    """
    return (
        user.person_id == person.person_id
        or is_holding_coordinator(user, person.company.holding_id)
        or user.person_id == person.support_id
    )


def is_need_person_trip_verification(user: User, person: Person) -> bool:
    """
    Return True if company country is Citizenship.KZ or user is holding coordinator

    :param user: user whose perms are being checked
    :param person: the person whose data the user reads
    :return:
    """
    return (
        person.company.country == Citizenship.KZ
        and not is_holding_coordinator(user, person.company.holding_id)
    )


async def has_trip_read_perm(user: User, trip: Trip, roles_map: LazyRolesMap) -> bool:
    """
    Check permission for read trip and return True if:
     - user is holding/yandex coordinator
     - or one of the participants
     - or one of person trip's managers
     - or one of participant's chiefs

    :param user: user whose perms are being checked
    :param trip: the trip whose data the user reads
    :param roles_map: roles_map of trip participants
    :return:
    """
    if is_holding_coordinator(user, trip.author.company.holding_id):
        return True
    if user.person_id == trip.author.person_id:
        return True
    if trip.person_trips is None:
        return False

    user_id = user.person_id
    for person_trip in trip.person_trips:
        participant_id = person_trip.person.person_id
        if (
            user_id == participant_id
            or user_id == person_trip.manager_id
            or is_support_for_person_trip(user, person_trip)
            or await roles_map.is_chief(user_id, participant_id)
            or await roles_map.is_approver(user_id, participant_id)
        ):
            return True
    return False


async def has_service_read_perm(
    user: User,
    service: Service,
    roles_map: LazyRolesMap,
    managers_map: LazyManagersMapByTrip,
) -> bool:
    """
    Check permission for read service

    :param user: the user whose perms are being checked
    :param service: the service whose data the user reads
    :param roles_map: roles_map of trip participants
    :param managers_map: managers of persons trips
    :return: has_person_perm or (author and not limited_access) or manager or chief
    """
    return (
        user.person_id == service.person_id
        or is_holding_coordinator(user, service.person.company.holding_id)
        or is_support_for_person_trip(user, service.person_trip)
        or (not user.is_limited_access and service.trip_author_id == user.person_id)
        or await roles_map.is_chief(user.person_id, service.person_id)
        or await roles_map.is_approver(user.person_id, service.person_id)
        or await managers_map.is_manager(user.person_id, service.trip_id)
    )


async def has_person_trip_read_perm(
    user: User,
    person_trip: PersonTrip,
    roles_map: LazyRolesMap,
) -> bool:
    """
    Check permission for read person trip

    :param user: the user whose perms are being checked
    :param person_trip: the person trip whose data the user reads
    :param roles_map: roles_map of trip participants
    :return: has_person_perm or (author and not limited_access) or manager or chief
    """
    return (
        user.person_id == person_trip.person_id
        or is_holding_coordinator(user, person_trip.person.company.holding_id)
        or (not user.is_limited_access and person_trip.trip_author_id == user.person_id)
        or is_support_for_person_trip(user, person_trip)
        or user.person_id == person_trip.manager_id
        or await roles_map.is_chief(user.person_id, person_trip.person_id)
        or await roles_map.is_approver(user.person_id, person_trip.person_id)
    )


async def check_user_perm_by_person(
    uow: UnitOfWork,
    user: User,
    person_id: int,
) -> None:
    """
    Check permissions (has_person_perm) user for person:
    This coroutine should be awaited (by default) in all places where the user get access to data
    if no other checks are performed

    :param uow: unit of work
    :param user: the user whose perms are being checked
    :param person_id: id of requested person's data
    """
    person = await uow.persons.get_person(person_id)
    if not has_person_perm(user, person):
        raise PermissionDenied(log_message=f'User has not read permission for person {person_id}')


async def has_transaction_read_perm(
    uow: UnitOfWork,
    user: User,
    transaction: Transaction,
    managers_map: LazyManagersMapByTrip,
) -> bool:
    """
    Check permission for read billing transaction

    :param uow: unit of work
    :param user: the user whose perms are being checked
    :param transaction: the transaction whose data the user reads
    :param managers_map: managers of persons trips
    :return: holding coordinator or manager
    """
    company = await uow.companies.get_company(transaction.company_id)
    return (
        is_holding_coordinator(user, company.holding_id)
        or await managers_map.is_manager(user.person_id, transaction.trip_id)
    )
