from dataclasses import asdict
import logging

from fastapi import APIRouter, Depends, HTTPException, Request

from intranet.trip.src.api.auth import (
    has_person_partial_perm,
    has_person_perm,
    has_person_trip_read_perm,
    has_service_read_perm,
    LazyRolesMap,
    LazyManagersMapByTrip,
)
from intranet.trip.src.api.depends import ZeroBasedPagination, get_cache, get_unit_of_work
from intranet.trip.src.api.schemas import PaginatedResponse
from intranet.trip.src.cache import Cache
from intranet.trip.src.config import settings
from intranet.trip.src.exceptions import PermissionDenied
from intranet.trip.src.lib.aeroclub import enums
from intranet.trip.src.lib.aeroclub.api import aeroclub
from intranet.trip.src.lib.aeroclub.models import (
    ACBonusCard,
    ACRailSearchResult,
    ACSearchRequestStatus,
    ACService,
    BusinessTrip,
    FilterParameter,
    Profile,
    Reference,
    SearchFilter,
    SearchRequest,
    SearchResult,
    SearchResultCount,
    SearchResultInList,
    TravellerDocument,
)
from intranet.trip.src.logic.aeroclub.documents import (
    fill_fio_in_passports,
    mask_passport_data,
    mask_traveller_document,
)
from intranet.trip.src.logic.aeroclub.reference import get_reference_list
from intranet.trip.src.logic.persons import PersonalDataGateway, get_personal_data_gateway
from intranet.trip.src.unit_of_work import UnitOfWork

logger = logging.getLogger(__name__)
router = APIRouter()


async def _check_service_perm(request: Request, order_id: int, service_id: int):
    acquire_kwargs = {'read_only': True} if settings.ENABLE_PYSCOPG2 else {}
    async with request.app.state.db.acquire(**acquire_kwargs) as conn:
        uow = UnitOfWork(conn)
        service = await uow.services.get_service_by_aeroclub_id(order_id, service_id)

        roles_map = LazyRolesMap(uow, person_ids=[service.person.person_id])
        managers_map = LazyManagersMapByTrip(uow, trip_ids=[service.trip_id])
        if not await has_service_read_perm(
                user=request.state.user,
                service=service,
                roles_map=roles_map,
                managers_map=managers_map,
        ):
            raise HTTPException(status_code=403, detail='not your service')


async def _check_person_trip_perm(request: Request, journey_id: int, trip_id: int):
    acquire_kwargs = {'read_only': True} if settings.ENABLE_PYSCOPG2 else {}
    async with request.app.state.db.acquire(**acquire_kwargs) as conn:
        uow = UnitOfWork(conn)
        person_trip = await uow.person_trips.get_person_trip_by_aeroclub_id(
            aeroclub_journey_id=journey_id,
            aeroclub_trip_id=trip_id,
        )

        roles_map = LazyRolesMap(uow, person_ids=[person_trip.person.person_id])
        if not await has_person_trip_read_perm(
            request.state.user, person_trip, roles_map,
        ):
            raise PermissionDenied(
                log_message=(
                    'User has not read permission for person_trip '
                    f'({person_trip.trip_id=}, {person_trip.person_id=})'
                ),
            )


@router.get(
    '/references',
    response_model=list[Reference],
    summary='Саджест по сущностям Аэроклуба (города, страны, отели и т.д.)',
)
async def reference_list(
    query: str,
    type: enums.ReferenceType,
    cache: Cache = Depends(get_cache),
):
    return await get_reference_list(query, type, cache)


@router.get(
    '/search/{search_request_id}/filterparameters',
    response_model=list[FilterParameter],
    summary='Получение доступных параметров фильтрации созданного поискового запроса',
)
async def search_filter_list(search_request_id: int):
    data = await aeroclub.get_search_filters(search_request_id)
    return data['data']


@router.get(
    '/search/{search_request_id}/status',
    response_model=ACSearchRequestStatus,
    summary='Получение статуса поискового запроса',
)
async def search_request_status(search_request_id: int) -> dict:
    data = await aeroclub.get_search_request(search_request_id)
    return {'status': data['data']['status']}


@router.get(
    '/search/{search_request_id}/details',
    response_model=SearchRequest,
    summary='Детализация объекта поискового запроса',
)
async def search_request_detail(search_request_id: int):
    data = await aeroclub.get_search_request(search_request_id)
    item = data['data']
    return dict(raw=item, **item)


@router.get(
    '/search/{search_request_id}',
    response_model=PaginatedResponse[SearchResultInList],
    summary='Результаты поискового запроса (все опции)',
)
async def search_result_list_deprecated(
    search_request_id: int,
    pagination=Depends(ZeroBasedPagination),
    search_filter=Depends(SearchFilter),
):
    params = asdict(search_filter)
    params['pageNumber'] = pagination.page
    params['maxCount'] = pagination.limit

    data = await aeroclub.get_search_results(search_request_id, params=params)
    results = [dict(raw=item, **item) for item in data['data']]
    return pagination.get_paginated_response(
        data=results,
        count=data['items_count'],
    )


@router.get(
    '/search/{search_request_id}/results/count',
    response_model=SearchResultCount,
    summary='Количество результатов поискового запроса (все опции)',
)
async def search_result_count(
    search_request_id: int,
    search_filter=Depends(SearchFilter),
):
    params = asdict(search_filter)
    data = await aeroclub.get_search_results_count(search_request_id, params=params)
    return data['data']


@router.get(
    '/search/{search_request_id}/options/{option_id}/results',
    response_model=PaginatedResponse[SearchResultInList],
    summary='Результаты одной опции поискового запроса',
)
async def search_option_result_list(
    search_request_id: int,
    option_id: int,
    pagination=Depends(ZeroBasedPagination),
    search_filter=Depends(SearchFilter),
):
    params = asdict(search_filter)
    params['pageNumber'] = pagination.page
    params['maxCount'] = pagination.limit
    data = await aeroclub.get_search_option_results(search_request_id, option_id, params=params)
    results = [dict(raw=item, **item) for item in data['data']]
    return pagination.get_paginated_response(
        data=results,
        count=data['items_count'],
    )


@router.get(
    '/search/{search_request_id}/options/{option_id}/results/count',
    response_model=SearchResultCount,
    summary='Количество результатов одной опции поискового запроса',
)
async def search_result_option_count(
    search_request_id: int,
    option_id: int,
    search_filter=Depends(SearchFilter),
):
    params = asdict(search_filter)
    data = await aeroclub.get_search_option_results_count(
        request_id=search_request_id,
        option_id=option_id,
        params=params,
    )
    return data['data']


@router.get(
    '/search/{search_request_id}/options/{option_id}',
    response_model=SearchResult,
    summary='Детальная информация одного варианта поискового запроса',
)
async def search_option_detail(
    search_request_id: int,
    option_id: int,
    key: str,
    detail_index: int = None,
):
    data = await aeroclub.get_search_option_details(
        request_id=search_request_id,
        option_id=option_id,
        key=key,
        detail_index=detail_index,
    )
    item = data['data']
    return dict(raw=item, **item)


@router.get(
    '/orders/{order_id}/services/{service_id}',
    response_model=ACService,
    summary='Детальная информация о услуге',
)
async def aeroclub_service_detail(request: Request, order_id: int, service_id: int):
    await _check_service_perm(request, order_id, service_id)

    data = await aeroclub.get_service(
        order_id=order_id,
        service_id=service_id,
    )
    return dict(raw=data, **data)


@router.get(
    '/orders/{order_id}/services/{service_id}/seats',
    response_model=ACRailSearchResult,
    summary='Доступные места ЖД-услуги',
)
async def aeroclub_service_seats(request: Request, order_id: int, service_id: int):
    await _check_service_perm(request, order_id, service_id)

    data = await aeroclub.get_free_seats(
        order_id=order_id,
        service_id=service_id,
    )
    data = data['data']
    return dict(raw=data, **data)


@router.get(
    path='/orders/{order_id}/services/{service_id}/allowed_documents',
    response_model=list[TravellerDocument],
    summary='Доступные документы для добавления в услугу',
)
async def allowed_document_list(
    request: Request,
    order_id: int,
    service_id: int,
    person_id: int,
    uow: UnitOfWork = Depends(get_unit_of_work(read_only=True)),
):
    pdg: PersonalDataGateway = await get_personal_data_gateway(uow)
    person = await pdg.get_person(person_id)
    profile_id = person.provider_profile_id

    if profile_id is None:
        raise HTTPException(status_code=404)
    await _check_service_perm(request, order_id, service_id)

    allowed_documents_list = await aeroclub.get_allowed_documents(
        order_id=order_id,
        service_id=service_id,
        profile_id=profile_id,
    )

    fill_fio_in_passports(allowed_documents_list, person)
    if request.state.user.person_id != person.person_id:
        mask_traveller_document(allowed_documents_list)
    return allowed_documents_list


@router.get(
    path='/person_profiles/{person_id}',
    response_model=Profile,
    summary='Данные профиля из Аэроклуба',
)
async def get_aeroclub_profile_by_person_id(
    request: Request,
    person_id: int,
    uow: UnitOfWork = Depends(get_unit_of_work(read_only=True)),
):
    user = request.state.user
    person = await uow.persons.get_person(person_id)
    roles_map = LazyRolesMap(uow, person_ids=[person_id])
    if not await has_person_partial_perm(user, person, roles_map):
        raise PermissionDenied(log_message='User has not partial permission for person')
    if person.provider_profile_id is None:
        raise PermissionDenied(log_message='provider_profile_id is empty')

    profile = await aeroclub.get_profile(person.provider_profile_id)

    if user.person_id != person.person_id:
        mask_passport_data(profile['documents'])

    return profile


@router.get(
    path='/person_profiles/{person_id}/bonus_cards',
    response_model=list[ACBonusCard],
    summary='Список бонусных карт профиля из Аэроклуба по нашему person_id',
)
async def get_aeroclub_profile_bonus_cards_by_person_id(
    request: Request,
    person_id: int,
    uow: UnitOfWork = Depends(get_unit_of_work(read_only=True)),
):
    person = await uow.persons.get_person(person_id)
    if not has_person_perm(request.state.user, person):
        raise PermissionDenied(log_message='User has not read permission for person')

    if person.provider_profile_id is None:
        raise PermissionDenied(log_message='provider_profile_id is empty')

    provider_profile_data = await aeroclub.get_profile(person.provider_profile_id)
    return provider_profile_data.get('bonus_cards', [])


@router.get(
    '/journeys/{journey_id}/trips/{trip_id}',
    response_model=BusinessTrip,
    summary='Детальная информация о персональной командировке (включая услуги)',
)
async def aeroclub_trip_detail(request: Request, journey_id: int, trip_id: int):
    await _check_person_trip_perm(request, journey_id, trip_id)

    return await aeroclub.get_trip(
        journey_id=journey_id,
        trip_id=trip_id,
    )
