import logging

from fastapi import APIRouter, Depends, Request, Response, HTTPException

from intranet.library.fastapi_csrf.src import csrf_exempt

from intranet.trip.src.api.depends import get_unit_of_work
from intranet.trip.src.api.schemas import PersonTripTrackerNotification
from intranet.trip.src.config import settings
from intranet.trip.src.exceptions import PermissionDenied
from intranet.trip.src.unit_of_work import UnitOfWork


logger = logging.getLogger(__name__)
router = APIRouter()


trip_queue = 'TRAVEL'


async def get_manager_id(
    uow: UnitOfWork,
    manager_login: str,
) -> int:
    if not manager_login:
        return None
    return await uow.persons.get_id_by_login(manager_login)


@router.post('/tracker/approve/', status_code=200, response_class=Response)
@csrf_exempt
async def approve_person_trip(
    request: Request,
    approve_notification: PersonTripTrackerNotification,
    uow: UnitOfWork = Depends(get_unit_of_work()),
):
    if request.state.user.login != settings.TRIP_ROBOT_LOGIN:
        logger.warning('Trying to approve person trip by %s', request.state.user.login)
        raise PermissionDenied(status_code=403)

    person_login = approve_notification.login
    staff_trip_uuid = approve_notification.staff_trip_uuid

    if person_login is not None and staff_trip_uuid is not None:
        person_id = await uow.persons.get_id_by_login(person_login)
        trip_id = await uow.trips.get_id_by_trip_uuid(staff_trip_uuid=staff_trip_uuid)

        person_trip = await uow.person_trips.get_detailed_person_trip(
            trip_id=trip_id,
            person_id=person_id,
        )
    else:
        person_trip = await uow.person_trips.get_person_trip_by_issue(
            issue_key=approve_notification.issue_key,
        )

    is_conf = person_trip.conf_details is not None
    if is_conf and approve_notification.issue_key.upper().startswith(trip_queue):
        raise HTTPException(status_code=400, detail='Forbidden')

    await uow.person_trips.update(
        person_trip.trip_id,
        person_trip.person_id,
        is_approved=True,
        manager_id=await get_manager_id(uow, approve_notification.manager_login),
    )
    logger.info(
        'person trip approved for trip_id=%s, person_id=%s. Ticket: %s',
        person_trip.trip_id,
        person_trip.person_id,
        approve_notification.issue_key,
    )


@router.post('/tracker/set_manager/', status_code=200, response_class=Response)
@csrf_exempt
async def set_person_trip_manager(
    request: Request,
    tracker_notification: PersonTripTrackerNotification,
    uow: UnitOfWork = Depends(get_unit_of_work()),
):
    if request.state.user.login != settings.TRIP_ROBOT_LOGIN:
        logger.warning('Trying to set manager by %s', request.state.user.login)
        raise PermissionDenied(status_code=403)

    person_login = tracker_notification.login
    staff_trip_uuid = tracker_notification.staff_trip_uuid

    if person_login is not None and staff_trip_uuid is not None:
        person_id = await uow.persons.get_id_by_login(person_login)
        trip_id = await uow.trips.get_id_by_trip_uuid(staff_trip_uuid=staff_trip_uuid)
    else:
        person_trip = await uow.person_trips.get_person_trip_by_issue(
            issue_key=tracker_notification.issue_key,
        )
        trip_id = person_trip.trip_id
        person_id = person_trip.person_id

    await uow.person_trips.update(
        trip_id,
        person_id,
        manager_id=await get_manager_id(uow, tracker_notification.manager_login),
    )
    logger.info(
        'manager %s set for person trip trip_id=%s, person_id=%s. Ticket: %s',
        tracker_notification.manager_login,
        trip_id,
        person_id,
        tracker_notification.issue_key,
    )
